// background.js - 使用直接下载方式，不占用服务器带宽
// 注意：Referer 头通过 rules.json 静态规则添加

// 🔥 从 manifest.json 读取版本号，确保版本号始终同步
const EXT_VERSION = chrome.runtime.getManifest().version;
const API_BASE_URL = "https://zhushou.xokx.top/api";

// 🔥 调试开关 - 生产环境设为 false
const DEBUG = false;
const log = DEBUG ? console.log.bind(console) : () => {};
const logError = DEBUG ? console.error.bind(console) : () => {};

// 🔥 版本检查：清除旧缓存
chrome.storage.local.get(['ext_version'], (result) => {
    // 🔥 添加 result 空值检查，防止多次刷新插件时报错
    if (result && result.ext_version !== EXT_VERSION) {
        log(`[版本更新] 从 ${result.ext_version || '未知'} 更新到 ${EXT_VERSION}，清除缓存...`);
        chrome.storage.local.set({ ext_version: EXT_VERSION });
        // 清除所有缓存
        if (self.caches) {
            caches.keys().then(names => {
                names.forEach(name => caches.delete(name));
            });
        }
    }
});

// 🔥 确保点击插件图标时打开侧边栏（Chrome 114+）
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true })
    .then(() => console.log('[侧边栏] 已设置为点击图标自动打开'))
    .catch((err) => console.error('[侧边栏] 设置失败:', err));

// 🔥 插件安装成功后打开说明页面
chrome.runtime.onInstalled.addListener((details) => {
    // 只在首次安装时打开说明页面，更新和浏览器升级时不打开
    if (details.reason === 'install') {
        chrome.tabs.create({
            url: 'https://zhushou.xokx.top/pages/manual.html'
        });
    }
});

let activeTasks = {};

// ============================================================
// 请求队列系统
// ============================================================

const QUEUE_CONFIG = {
    MAX_CONCURRENT: 10,
    MAX_RETRIES: 2,
    RETRY_DELAY: 3000,
    QUEUE_STORAGE_KEY: 'request_queue_state'
};

class RequestQueue {
    constructor(config) {
        this.maxConcurrent = config.MAX_CONCURRENT;
        this.maxRetries = config.MAX_RETRIES;
        this.retryDelay = config.RETRY_DELAY;
        this.storageKey = config.QUEUE_STORAGE_KEY;

        this.queue = {
            vip: [],
            normal: []
        };
        this.processing = {};

        this.restoreState();
    }

    async enqueue(task) {
        const taskObj = {
            id: generateTaskId(),
            type: task.type,
            params: task.params,
            isVip: task.isVip || false,
            retryCount: 0,
            addTime: Date.now(),
            senderTabId: task.senderTabId
        };

        const targetQueue = taskObj.isVip ? this.queue.vip : this.queue.normal;
        targetQueue.push(taskObj);

        this.saveState();
        this.process();

        const queueName = taskObj.isVip ? 'VIP队列' : '普通队列';
        const queueLength = targetQueue.length;
        log(`[队列] 任务已加入${queueName} | 任务ID: ${taskObj.id} | 队列长度: ${queueLength}`);

        return taskObj.id;
    }

    async process() {
        const currentProcessing = Object.keys(this.processing).length;
        const availableSlots = this.maxConcurrent - currentProcessing;

        if (availableSlots <= 0) {
            log(`[队列] 已达到最大并发数 (${this.maxConcurrent})，等待中...`);
            return;
        }

        let task = this.queue.vip.shift() || this.queue.normal.shift();

        if (!task) {
            return;
        }

        this.processing[task.id] = task;
        this.saveState();

        log(`[队列] 开始处理任务 | ID: ${task.id} | 类型: ${task.type} | VIP: ${task.isVip}`);

        this.executeTask(task).catch(error => {
            logError(`[队列] 任务执行失败 | ID: ${task.id} | 错误:`, error);
        });

        if (availableSlots > 1) {
            this.process();
        }
    }

    async executeTask(task) {
        try {
            let result;

            switch (task.type) {
                case 'transcribeAudio':
                    result = await this.executeTranscribe(task);
                    break;
                case 'downloadMedia':
                    result = await this.executeDownload(task);
                    break;
                default:
                    throw new Error(`未知的任务类型: ${task.type}`);
            }

            delete this.processing[task.id];
            this.saveState();

            log(`[队列] 任务完成 | ID: ${task.id}`);

            this.process();

            return result;

        } catch (error) {
            await this.handleTaskError(task, error);
        }
    }

    async executeTranscribe(task) {
        const { videoUrl } = task.params;

        // 初始化任务状态
        notifyTaskUpdate(task.id, "loading", "正在解析地址...", videoUrl);

        await startAsyncTranscription(videoUrl, task.id, task.senderTabId);

        return { success: true };
    }

    async executeDownload(task) {
        const { videoUrl, pageAuthor, pageTitle, type } = task.params;

        const result = await handleDownload(videoUrl, pageAuthor, pageTitle, type);

        if (result.status === 'error') {
            throw new Error(result.msg);
        }

        return result;
    }

    async handleTaskError(task, error) {
        delete this.processing[task.id];

        const shouldRetry = this.shouldRetry(task, error);

        if (shouldRetry) {
            task.retryCount++;

            log(`[队列] 重试任务 | ID: ${task.id} | 第 ${task.retryCount} 次重试`);

            setTimeout(() => {
                const targetQueue = task.isVip ? this.queue.vip : this.queue.normal;
                targetQueue.push(task);
                this.saveState();
                this.process();
            }, this.retryDelay);

        } else {
            logError(`[队列] 任务最终失败 | ID: ${task.id} | 错误: ${error.message}`);

            if (task.type === 'transcribeAudio') {
                let errorMsg = error.message;
                if (error.name === 'AbortError') {
                    errorMsg = '请求超时，请重试';
                } else if (errorMsg.toLowerCase().includes('failed to fetch')) {
                    errorMsg = '网络请求失败，请检查网络';
                }

                if (errorMsg.includes('今日免费额度')) {
                    return;
                }

                notifyTaskUpdate(task.id, 'error', errorMsg);
            }

            this.saveState();
            this.process();
        }
    }

    shouldRetry(task, error) {
        if (task.retryCount >= this.maxRetries) {
            return false;
        }

        if (error.message && error.message.includes('今日免费额度')) {
            return false;
        }

        if (error.message && error.message.includes('需要 VIP 权限')) {
            return false;
        }

        if (error.message && error.message.includes('您的调用过于频繁')) {
            return false;
        }

        if (error.message && error.message.includes('下载过于频繁')) {
            return true;
        }

        const retryableErrors = [
            'Failed to fetch',
            'Network error',
            'Timeout',
            '超时',
            '网络'
        ];

        return retryableErrors.some(err =>
            error.message && error.message.includes(err)
        );
    }

    saveState() {
        chrome.storage.local.set({
            [this.storageKey]: {
                queue: this.queue,
                processing: this.processing
            }
        });
    }

    restoreState() {
        chrome.storage.local.get([this.storageKey], (result) => {
            if (result && result[this.storageKey]) {
                const state = result[this.storageKey];
                this.queue = state.queue || { vip: [], normal: [] };
                this.processing = state.processing || {};

                for (const [taskId, task] of Object.entries(this.processing)) {
                    log(`[队列] 恢复中断的任务 | ID: ${taskId}`);
                    this.executeTask(task);
                }

                log(`[队列] 状态已恢复 | VIP队列: ${this.queue.vip.length} | 普通队列: ${this.queue.normal.length}`);
            }
        });
    }

    getStatus() {
        return {
            vipQueueLength: this.queue.vip.length,
            normalQueueLength: this.queue.normal.length,
            processingCount: Object.keys(this.processing).length,
            maxConcurrent: this.maxConcurrent
        };
    }
}

const requestQueue = new RequestQueue(QUEUE_CONFIG);

setInterval(() => {
    const status = requestQueue.getStatus();
    log(`[队列状态] VIP: ${status.vipQueueLength} | 普通: ${status.normalQueueLength} | 处理中: ${status.processingCount}/${status.maxConcurrent}`);
}, 60 * 1000);

const TASK_TIMEOUT = 5 * 60 * 1000;
setInterval(() => {
    const now = Date.now();
    let cleanedCount = 0;

    for (const [taskId, task] of Object.entries(activeTasks)) {
        if (task.startTime && now - task.startTime > TASK_TIMEOUT) {
            log(`[清理过期任务] ${taskId}，运行时长: ${Math.round((now - task.startTime) / 1000)}秒`);
            delete activeTasks[taskId];
            cleanedCount++;
        }
    }

    if (cleanedCount > 0) {
        chrome.storage.local.set({ 'active_tasks': activeTasks });
        log(`[清理过期任务] 已清理 ${cleanedCount} 个过期任务`);
    }
}, 60 * 1000);

function generateTaskId() { return 'task_' + Date.now() + '_' + Math.floor(Math.random() * 1000); }

function checkVipStatus() {
    return new Promise(resolve => {
        chrome.storage.local.get(['user_info', 'token'], (res) => {
            // 🔥 首先检查是否登录（token是否存在）
            if (!res.token) {
                resolve(false);
                return;
            }

            const userInfo = res.user_info;

            // 未登录或无用户信息
            if (!userInfo) {
                resolve(false);
                return;
            }

            // 检查 VIP 状态和过期时间
            const isVip = userInfo.is_vip === true || userInfo.vip_status === 'active';
            const vipExpiresAt = userInfo.vip_expires_at || userInfo.expires_at;

            if (!isVip) {
                resolve(false);
                return;
            }

            // 检查是否过期
            if (vipExpiresAt) {
                const now = Math.floor(Date.now() / 1000); // 当前时间戳（秒）
                if (now > vipExpiresAt) {
                    resolve(false); // 已过期
                    return;
                }
            }

            resolve(true); // 是有效的 VIP
        });
    });
}

function getToken() {
    return new Promise(resolve => chrome.storage.local.get(['token'], res => resolve(res.token)));
}

// 🔥 处理IP不匹配错误：自动退出登录
function handleIpMismatch() {
    log('[安全检测] 检测到IP地址变更，正在退出登录...');
    // 清除登录状态
    chrome.storage.local.remove(['token', 'user_info'], () => {
        log('[安全检测] 已清除登录状态');
    });
}

function checkGuestLimit(type) {
    return new Promise(resolve => {
        chrome.storage.local.get(['guest_usage'], (res) => {
            let usage = res.guest_usage || { date: '', video: 0, audio: 0, copy: 0 };
            const currentDate = new Date().toISOString().slice(0, 10); 
            
            if (usage.date !== currentDate) {
                usage = { date: currentDate, video: 0, audio: 0, copy: 0 };
                chrome.storage.local.set({ guest_usage: usage });
            }
            resolve(usage[type] < 5); // 5次限制
        });
    });
}

function incrementGuestUsage(type) {
    chrome.storage.local.get(['guest_usage'], (res) => {
        let usage = res.guest_usage || { date: new Date().toISOString().slice(0, 10), video: 0, audio: 0, copy: 0 };
        usage[type] = (usage[type] || 0) + 1;
        chrome.storage.local.set({ guest_usage: usage });
    });
}

// [修改] 默认超时改为 120 秒
async function fetchWithTimeout(resource, options = {}) {
    const { timeout = 120000 } = options; 
    const controller = new AbortController();
    const id = setTimeout(() => controller.abort(), timeout);
    try {
        const response = await fetch(resource, { ...options, signal: controller.signal });
        clearTimeout(id);
        return response;
    } catch (error) {
        clearTimeout(id);
        throw error;
    }
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "openSidePanel") {
        if (sender.tab && sender.tab.id) {
            chrome.sidePanel.open({ tabId: sender.tab.id }).catch(() => {
                // 静默处理，不输出日志
            });
        }
        return true;
    }

    if (request.action === "checkLimit") {
        const type = request.type;
        chrome.storage.local.get(['user_info', 'token'], async (res) => {
            // 🔥 检查是否登录
            const isLoggedIn = !!res.token;

            if (!isLoggedIn) {
                // 未登录，打开侧边栏并触发登录框
                if (sender.tab && sender.tab.id) {
                    // 设置标志位，让侧边栏自动弹出登录框
                    chrome.storage.local.set({ _openLoginModal: true });
                    chrome.sidePanel.open({ tabId: sender.tab.id }).catch(() => {});
                }
                sendResponse({ allowed: false, isLoggedIn: false, msg: "请先登录/注册后使用" });
                return;
            }

            // 已登录，直接允许
            sendResponse({ allowed: true, isLoggedIn: true });
        });
        return true;
    }

    // 🚀 使用队列系统处理音频转录请求
    if (request.action === "transcribeAudio") {
        // 异步检查 VIP 状态并加入队列
        checkVipStatus().then(isVip => {
            return requestQueue.enqueue({
                type: 'transcribeAudio',
                params: {
                    videoUrl: request.videoUrl
                },
                isVip: isVip,
                senderTabId: sender.tab?.id || null
            });
        }).then(taskId => {
            sendResponse({ status: "queued", taskId: taskId });
        }).catch(error => {
            sendResponse({ status: "error", msg: error.message });
        });

        return true;
    }

    // 🚀 使用队列系统处理媒体下载请求
    if (request.action === "downloadMedia") {
        // 异步检查 VIP 状态并加入队列
        checkVipStatus().then(isVip => {
            return requestQueue.enqueue({
                type: 'downloadMedia',
                params: {
                    videoUrl: request.videoUrl,
                    pageAuthor: request.pageAuthor,
                    pageTitle: request.pageTitle,
                    type: request.type
                },
                isVip: isVip,
                senderTabId: sender.tab?.id || null
            });
        }).then(taskId => {
            sendResponse({ status: "queued", taskId: taskId });
        }).catch(error => {
            sendResponse({ status: "error", msg: error.message });
        });

        return true;
    }

    // ==================== 分屏弹窗独立窗口相关 ====================
    // 处理来自独立窗口的保存请求
    if (request.action === "saveCloudRecordFromSplitModal") {
        handleSaveCloudRecord(request).then(result => {
            sendResponse(result);
        }).catch(error => {
            sendResponse({ success: false, error: error.message });
        });
        return true;
    }

    // 处理独立窗口关闭通知
    if (request.action === "splitModalWindowClosed") {
        console.log('[分屏弹窗] 窗口已关闭');
        sendResponse({ received: true });
        return true;
    }

    // ==================== 提取记录弹窗独立窗口相关 ====================
    // 处理来自提取记录弹窗的保存请求
    if (request.action === "saveExtractionRecord") {
        handleSaveExtractionRecord(request).then(result => {
            sendResponse(result);
        }).catch(error => {
            sendResponse({ success: false, error: error.message });
        });
        return true;
    }

    // 处理提取记录弹窗关闭通知
    if (request.action === "extractionModalWindowClosed") {
        console.log('[提取记录弹窗] 窗口已关闭');
        sendResponse({ received: true });
        return true;
    }

    // ==================== 改写历史弹窗独立窗口相关 ====================
    // 处理来自改写历史弹窗的保存请求
    if (request.action === "saveRewriteHistoryRecord") {
        handleSaveRewriteHistoryRecord(request).then(result => {
            sendResponse(result);
        }).catch(error => {
            sendResponse({ success: false, error: error.message });
        });
        return true;
    }
});

function notifyTaskUpdate(taskId, status, text) {
    if (status === 'loading' || status === 'error') {
        activeTasks[taskId] = { status, text, startTime: Date.now() };
    } else {
        delete activeTasks[taskId];
    }
    chrome.storage.local.set({ 'active_tasks': activeTasks });
    chrome.runtime.sendMessage({ action: "taskUpdate", taskId, status, data: text }, () => {
        void chrome.runtime.lastError; // 显式检查，侧边栏关闭时忽略错误
    });
}

async function startAsyncTranscription(pageUrl, taskId, senderTabId) {
    // 动态提示信息数组 - 模拟专业的AI处理流程
    const progressMessages = [
        "• 正在分析音频波形...",
        "• 正在进行语音识别...",
        "• 正在智能断句分词...",
        "• 正在检查错别字...",
        "• 正在进行语气分析...",
        "• 正在优化标点符号...",
        "• 正在进行智能润色...",
        "• 正在生成最终结果..."
    ];

    let messageIndex = 0;
    let progressInterval = null;

    try {
        const token = await getToken();
        if (!token && !(await checkGuestLimit('copy'))) throw new Error("游客额度已用完");

        const sources = await getAiMediaSources(pageUrl, token);
        if (!sources) throw new Error("未能提取到媒体地址");

        if (!token) incrementGuestUsage('copy');

        // 🔥 启动动态提示
        notifyTaskUpdate(taskId, "loading", progressMessages[0]);
        messageIndex = 0;
        progressInterval = setInterval(() => {
            messageIndex = (messageIndex + 1) % progressMessages.length;
            notifyTaskUpdate(taskId, "loading", progressMessages[messageIndex]);
        }, 4000); // 每4秒更新一次提示

        const text = await getTranscriptionFromServer(sources.audio, sources.video, token, pageUrl);
        const cleanText = cleanTextStr(text);

        // 清除定时器
        if (progressInterval) {
            clearInterval(progressInterval);
            progressInterval = null;
        }

        await saveHistory(cleanText);
        notifyTaskUpdate(taskId, "success", cleanText);

    } catch (error) {
        // 清除定时器
        if (progressInterval) {
            clearInterval(progressInterval);
            progressInterval = null;
        }

        // 🔥 检查错误消息，判断是否为额度超限错误（使用后端返回的消息）
        const isLimitExceeded = error.message && (
            error.message.includes("免费额度已用完") ||
            error.message.includes("免费额度")
        );

        let showMsg = error.message;

        // 🔥 处理免费用户额度超限错误（直接使用后端返回的消息）
        if (isLimitExceeded) {
            // 🔥 使用保存的 tabId 发送消息到 content script
            if (senderTabId) {
                // 🔥 先验证 tab 是否还存在
                chrome.tabs.get(senderTabId, (tab) => {
                    if (chrome.runtime.lastError) {
                        return;
                    }
                    chrome.tabs.sendMessage(senderTabId, {
                        action: "showLimitExceededToast",
                        message: showMsg,  // 直接使用后端返回的消息
                        upgradeUrl: "/pages/pay.html"
                    }, (response) => {
                        // 静默处理，不输出日志
                    });
                });
            } else {
                // 降级方案：尝试使用 chrome.tabs.query
                chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: "showLimitExceededToast",
                            message: showMsg,  // 直接使用后端返回的消息
                            upgradeUrl: "/pages/pay.html"
                        }, (response) => {
                            // 静默处理，不输出日志
                        });
                    }
                });
            }

            // 🔥 额度超限错误不在侧边栏显示红色失败
            // 删除任务并通知侧边栏移除loading状态
            delete activeTasks[taskId];
            chrome.storage.local.set({ 'active_tasks': activeTasks });
            // 发送成功状态（空内容）来移除loading状态，不显示结果块
            chrome.runtime.sendMessage({ action: "taskUpdate", taskId, status: "success", data: "" }, () => {
                void chrome.runtime.lastError; // 显式检查，侧边栏关闭时忽略错误
            });
            return;
        } else {
            if (error.isVipRequired) {
                showMsg = error.message || "此功能需要 VIP 权限";
            }
            else if (error.name === 'AbortError') showMsg = "请求超时，请重试";
            else if (showMsg.toLowerCase().includes('failed to fetch')) showMsg = "网络请求失败，请检查网络";

            // 其他错误正常显示
            notifyTaskUpdate(taskId, "error", showMsg);
        }
    }
}

async function handleDownload(pageUrl, pageAuthor, pageTitle, type) {
    log('[handleDownload] 开始下载', { pageUrl, pageAuthor, pageTitle, type });
    try {
        const token = await getToken();
        log('[handleDownload] Token:', token ? '有' : '无');
        if (!token) {
            if (!(await checkGuestLimit(type))) return { status: "error", msg: "游客额度已用完" };
        }

        // 生成文件名：(作者)_标题_时间戳
        const safeAuthor = safeFileName(pageAuthor || "Unknown");
        const safeTitle = safeFileName(pageTitle || "Video");
        const timeHash = Date.now().toString().slice(-6);
        const fileExt = type === "audio" ? ".mp3" : ".mp4";
        const filename = `(${safeAuthor})_${safeTitle}_${timeHash}${fileExt}`;

        // 🔥 使用 /download-url 接口获取直链（不占用服务器带宽）
        const resolveUrl = `${API_BASE_URL}/download-url?url=${encodeURIComponent(pageUrl)}&type=${type}`;
        const headers = token ? { "Authorization": `Bearer ${token}` } : {};

        const resolveRes = await fetchWithTimeout(resolveUrl, {
            method: "GET",
            headers: headers,
            timeout: 60000
        });

        const resolveData = await resolveRes.json();

        // 🔥 处理401错误（账号认证失败）
        if (resolveRes.status === 401) {
            handleIpMismatch();
            return { status: "error", msg: resolveData.error || "账号已在其他设备登录" };
        }

        // 🔥 处理 429 错误（高频下载限制）
        if (resolveRes.status === 429) {
            const errorMsg = resolveData.message || resolveData.error || "下载过于频繁，请稍后再试";
            return { status: "error", msg: errorMsg, isRateLimit: true };
        }

        if (!resolveRes.ok || resolveData.code !== 200) {
            throw new Error(resolveData.error || "获取下载链接失败");
        }

        const directUrl = resolveData.download_url;
        if (!directUrl) {
            throw new Error(`未找到可下载的${type === 'audio' ? '音频' : '视频'}资源`);
        }

        log('[下载] 直链URL:', directUrl.substring(0, 80) + '...');

        // 🔥 直接下载 CDN URL，由 declarativeNetRequest 添加 Referer 头
        chrome.downloads.download({
            url: directUrl,
            filename: filename,
            conflictAction: "uniquify",
            saveAs: false
        }, (downloadId) => {
            if (chrome.runtime.lastError) {
                logError('[下载错误]', chrome.runtime.lastError.message);
            } else {
                log('[下载开始] ID:', downloadId);
                const onDownloadChanged = (delta) => {
                    if (delta.id === downloadId && delta.state) {
                        if (delta.state.current === 'complete') {
                            log('[下载完成] ID:', downloadId);
                            chrome.downloads.onChanged.removeListener(onDownloadChanged);
                        } else if (delta.state.current === 'interrupted') {
                            chrome.downloads.search({ id: downloadId }, (results) => {
                                if (results[0]) {
                                    logError('[下载失败]', results[0].error, results[0].filename);
                                }
                            });
                            chrome.downloads.onChanged.removeListener(onDownloadChanged);
                        }
                    }
                };
                chrome.downloads.onChanged.addListener(onDownloadChanged);
            }
        });

        if (!token) incrementGuestUsage(type);
        return { status: "success" };
    } catch (e) {
        // 静默处理，不输出日志
        return { status: "error", msg: e.message };
    }
}

async function getAiMediaSources(pageUrl, token) {
    try {
        const res = await fetchWithTimeout(`${API_BASE_URL}/resolve`, {
            method: "POST",
            headers: { "Content-Type": "application/json", ...(token ? { "Authorization": `Bearer ${token}` } : {}) },
            body: JSON.stringify({ url: pageUrl, type: 'copy' }),
            timeout: 120000
        });
        const json = await res.json();

        // 🔥 处理401错误（账号认证失败）
        if (res.status === 401) {
            handleIpMismatch();
            throw new Error(json.error || "账号已在其他设备登录");
        }

        // 🔥 处理 403 错误（免费用户额度用完等）
        if (res.status === 403) {
            const limitError = new Error(json.message || json.error || "需要 VIP 权限");
            limitError.isLimitExceeded = true;
            limitError.displayMessage = json.message || json.error;
            limitError.upgradeUrl = json.upgradeUrl || "/pages/pay.html";
            throw limitError;
        }

        if (!res.ok) throw new Error(json.error || "后端请求拒绝");
        if (json.code == 200 && json.data) {
            let audio = json.data.music?.play_url || json.data.music?.url || json.data.music_url;
            let video = json.data.work_url || json.data.url || json.data.play_url;
            if (!audio) audio = video;
            return {
                audio: audio ? audio.replace(/^http:\/\//i, 'https://') : null,
                video: video ? video.replace(/^http:\/\//i, 'https://') : null,
                // 同时返回 API 中的作者和标题
                apiAuthor: json.data.author || json.data.nickname || json.data.author_name || json.data.user?.nickname || null,
                apiTitle: json.data.title || json.data.desc || json.data.desc_text || json.data.title_text || json.data.caption || null
            };
        }
        return null;
    } catch (e) { throw new Error(e.message || "链接解析服务异常"); }
}

async function getTranscriptionFromServer(audioUrl, videoUrl, token, pageUrl = '') {
    const res = await fetchWithTimeout(`${API_BASE_URL}/transcribe`, {
        method: "POST",
        headers: { "Content-Type": "application/json", ...(token ? { "Authorization": `Bearer ${token}` } : {}) },
        body: JSON.stringify({ fileUrl: audioUrl, videoUrl: videoUrl, pageUrl: pageUrl }),
        timeout: 180000
    });
    const data = await res.json();

    // 🔥 处理401错误（账号认证失败）
    if (res.status === 401) {
        handleIpMismatch();
        throw new Error(data.error || "账号已在其他设备登录");
    }

    // 🔥 处理 403 错误（免费用户额度用完等）- 提示文案由后端统一管理
    if (res.status === 403) {
        const limitError = new Error(data.message || data.error || "需要 VIP 权限");
        limitError.isLimitExceeded = true;

        // 🔥 后端统一管理提示信息，前端直接使用
        // 后端返回的数据结构：
        // {
        //   error: "错误标题",
        //   message: "详细的多行提示信息（后端统一管理）",
        //   upgradeUrl: "/pages/pay.html"
        // }
        limitError.displayMessage = data.message || data.error;
        limitError.upgradeUrl = data.upgradeUrl || "/pages/pay.html";

        throw limitError;
    }

    if (!res.ok) throw new Error(data.error || "转录失败");
    return data.text || "";
}

function saveHistory(text) {
    return new Promise((resolve) => {
        chrome.storage.local.get({ ai_history: [] }, (result) => {
            const list = result.ai_history || [];
            const now = new Date();
            const timeStr = `${(now.getMonth()+1).toString().padStart(2,'0')}-${now.getDate().toString().padStart(2,'0')} ${now.getHours().toString().padStart(2,'0')}:${now.getMinutes().toString().padStart(2,'0')}`;
            list.push({ content: text, time: timeStr });
            chrome.storage.local.set({ 'ai_history': list }, resolve);
        });
    });
}

function cleanTextStr(text) {
    if (!text) return "";
    // 移除特殊标记（分段功能已移至后端）
    return text.replace(/<\|.*?\|>/g, "").trim();
}
function safeFileName(str) {
    if (!str) return "Unknown";
    let safe = str.replace(/[\\/:*?"<>|]/g, " ");
    safe = safe.replace(/[\r\n]/g, "");

    // 🔥 移除音频/视频格式关键词（使用单词边界确保精确匹配）
    safe = safe.replace(/\b(mp3|mp4|wav|flv|avi|mov|mkv)\b/gi, "");
    safe = safe.replace(/\b(音频|视频|音乐|歌曲|高质量|HQ|HD)\b/gi, "");

    // 🔥 移除按钮文字和操作提示
    const ignoreKeywords = [
        '文案提取', '音频下载', '视频下载', '抢首评',
        '智能', '倍速', '大家都', '点赞', '评论',
        '分享', '收藏', '转发', '关注'
    ];
    for (const kw of ignoreKeywords) {
        safe = safe.replace(new RegExp(kw, 'g'), '');
    }

    // 移除多余的空格和符号
    safe = safe.replace(/\s+/g, " ").trim();
    safe = safe.replace(/^[^\u4e00-\u9fa5a-zA-Z0-9_\s（(【】）】）]+/g, ''); // 移除开头的非文字符号

    return safe.substring(0, 50).trim();
}

// ==================== 分屏弹窗独立窗口 - 保存云端记录 ====================
/**
 * 处理来自独立分屏窗口的保存请求
 */
async function handleSaveCloudRecord(request) {
    const { recordId, originalText, rewriteText } = request;

    // 获取 token
    const token = await getToken();
    if (!token) {
        return { success: false, error: "请先登录" };
    }

    if (!rewriteText || rewriteText.trim().length === 0) {
        return { success: false, error: "改写内容不能为空" };
    }

    try {
        const res = await fetch(`${API_BASE_URL}/user/rewrite/update/${recordId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                originalText: originalText,
                rewriteText: rewriteText
            })
        });

        if (res.ok) {
            // 通知 panel 刷新云端记录列表
            chrome.runtime.sendMessage({ action: 'refreshCloudRecords' }, () => {
                void chrome.runtime.lastError; // 忽略错误
            });

            return { success: true, message: "保存成功" };
        } else {
            const data = await res.json();
            return { success: false, error: data.error || "保存失败" };
        }
    } catch (e) {
        logError('[保存云端记录] Error:', e);
        return { success: false, error: "网络错误" };
    }
}

// ==================== 提取记录弹窗独立窗口 - 保存提取记录 ====================
/**
 * 处理来自提取记录弹窗的保存请求
 * 保存修改后的提取记录到本地存储
 */
async function handleSaveExtractionRecord(request) {
    const { index, content } = request;

    if (!content || content.trim().length === 0) {
        return { success: false, error: "内容不能为空" };
    }

    try {
        // 获取当前历史记录
        const result = await new Promise(resolve => {
            chrome.storage.local.get({ ai_history: [] }, resolve);
        });

        const items = result.ai_history || [];

        // 更新指定索引的记录
        if (index >= 0 && index < items.length) {
            if (typeof items[index] === 'string') {
                items[index] = content;
            } else {
                items[index].content = content;
            }

            // 保存回本地存储
            await new Promise(resolve => {
                chrome.storage.local.set({ ai_history: items }, resolve);
            });

            // 通知 panel 刷新历史记录列表
            chrome.runtime.sendMessage({ action: 'refreshExtractionRecords', items: items }, () => {
                void chrome.runtime.lastError; // 忽略错误
            });

            return { success: true, message: "保存成功" };
        } else {
            return { success: false, error: "记录不存在" };
        }
    } catch (e) {
        logError('[保存提取记录] Error:', e);
        return { success: false, error: "保存失败" };
    }
}

// ==================== 改写历史弹窗独立窗口 - 保存改写记录 ====================
/**
 * 处理来自改写历史弹窗的保存请求
 * 保存修改后的改写记录到本地存储
 */
async function handleSaveRewriteHistoryRecord(request) {
    const { originalText, newRewriteText, recordIndex } = request;

    if (!newRewriteText || newRewriteText.trim().length === 0) {
        return { success: false, error: "内容不能为空" };
    }

    try {
        // 获取当前历史记录
        const result = await new Promise(resolve => {
            chrome.storage.local.get({ ai_history: [] }, resolve);
        });

        const items = result.ai_history || [];

        // 查找匹配的历史项（通过原文匹配）
        const itemIndex = items.findIndex(item => {
            const content = typeof item === 'string' ? item : item.content;
            return content === originalText || content === (originalText || '').trim();
        });

        if (itemIndex !== -1 && items[itemIndex].rewrite && items[itemIndex].rewrite.length > 0) {
            // 更新指定索引的改写记录
            // recordIndex 是相对于 rewrite 数组的索引
            items[itemIndex].rewrite[recordIndex] = {
                text: newRewriteText,
                time: items[itemIndex].rewrite[recordIndex].time
            };

            // 保存回本地存储
            await new Promise(resolve => {
                chrome.storage.local.set({ ai_history: items }, resolve);
            });

            // 通知 panel 刷新历史记录列表
            chrome.runtime.sendMessage({ action: 'refreshExtractionRecords', items: items }, () => {
                void chrome.runtime.lastError; // 忽略错误
            });

            return { success: true, message: "保存成功" };
        } else {
            return { success: false, error: "记录不存在" };
        }
    } catch (e) {
        logError('[保存改写记录] Error:', e);
        return { success: false, error: "保存失败" };
    }
}