// content.js - v4.88 统一醒目渐变弹窗样式

let lastClickTime = 0;
const TEXT_VIDEO = "视频下载";
const TEXT_AUDIO = "音频下载";
const TEXT_COPY  = "文案提取";
// 使用服务器上的 download.png 图片
const ICON_URL = "https://zhushou.xokx.top/assets/download.png";
const DOWNLOAD_ICON_URL = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='white' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4M7 10l5 5 5-5M12 15V3'/%3E%3C/svg%3E";

const processingSet = new Set();

// 🔥 清除可能残留的旧按钮
document.querySelectorAll('.dy-tool-container').forEach(btn => btn.remove());

initPlugin();

function initPlugin() {
    setInterval(() => injectButton(), 1000);
}

function injectButton() {
    // 🔥 处理首页等常规页面
    const anchors = document.querySelectorAll(
        '[data-e2e="video-player-digg"], ' +
        '.xg-video-player-digg, ' +
        '[data-e2e="browse-video-card-digg"]'
    );

    anchors.forEach(anchor => {
        let sidebar = anchor.closest('.xg-right-grid') || anchor.parentNode?.parentNode;
        if (!sidebar) return;
        if (sidebar.querySelector('.dy-tool-container')) return;
        injectSingleButton(anchor, sidebar);
    });

    // 🔥 特殊处理详情页（直接打开视频链接的页面）
    // 需要单独处理，因为这类页面没有常规的 data-e2e 标记
    const detailContainer = document.querySelector('.fN2jqmuV') || document.querySelector('.bm6Yr1Fm');
    if (detailContainer && !detailContainer.querySelector('.dy-tool-container')) {
        const anchor = detailContainer.querySelector('.fcEX2ARL');
        if (anchor) {
            injectSingleButton(anchor, detailContainer, true);
        }
    }
}

function injectSingleButton(anchor, sidebar, useAppend = false) {

        const container = document.createElement('div');
        container.className = 'dy-tool-container';

        container.addEventListener('click', (e) => e.stopPropagation());
        container.addEventListener('dblclick', (e) => e.stopPropagation());

        container.innerHTML = `
            <div class="dy-menu">
                <div class="dy-pill-btn copy">${TEXT_COPY}</div>
                <div class="dy-pill-btn audio">${TEXT_AUDIO}</div>
                <div class="dy-pill-btn video">${TEXT_VIDEO}</div>
            </div>
            <div class="dy-main-btn" title="鼠标悬停查看菜单">
                <img src="${ICON_URL}" class="dy-btn-img dy-icon-static" />
                <img src="${DOWNLOAD_ICON_URL}" class="dy-btn-img dy-icon-loading" />
            </div>
        `;

        const btnVideo = container.querySelector('.video');
        const btnAudio = container.querySelector('.audio');
        const btnCopy  = container.querySelector('.copy');
        const mainBtn  = container.querySelector('.dy-main-btn');

        btnVideo.onclick = (e) => {
            e.stopPropagation();
            handleTask(e, anchor, "video", mainBtn);
        };

        btnAudio.onclick = (e) => {
            e.stopPropagation();
            handleTask(e, anchor, "audio", mainBtn);
        };

        btnCopy.onclick = (e) => {
            e.stopPropagation();
            handleTask(e, anchor, "copy", mainBtn);
        };

        // 点击主按钮主动打开侧边栏
        mainBtn.onclick = (e) => {
            e.stopPropagation();
            e.preventDefault();
            chrome.runtime.sendMessage({ action: "openSidePanel" });
        };

        try {
            if (useAppend) {
                sidebar.appendChild(container);
            } else {
                sidebar.prepend(container);
            }
        } catch (err) {
            // 静默处理错误
        }
}

function toggleLoading(btn, isLoading) {
    if (!btn) return;

    const container = btn.closest('.dy-tool-container');
    if (!container) return;

    const staticIcon = container.querySelector('.dy-icon-static');
    const loadingIcon = container.querySelector('.dy-icon-loading');

    if (isLoading) {
        btn.classList.add('loading');
        if (staticIcon) staticIcon.style.display = 'none';
        if (loadingIcon) loadingIcon.style.display = 'block';
    } else {
        btn.classList.remove('loading');
        if (staticIcon) staticIcon.style.display = 'block';
        if (loadingIcon) loadingIcon.style.display = 'none';
    }
}

function handleTask(e, anchor, type, mainBtn) {
    e.stopPropagation();
    e.stopImmediatePropagation();
    e.preventDefault();

    // 防抖
    const now = Date.now();
    if (now - lastClickTime < 1500) {
        showToast("请慢一点点击，防止接口拥堵");
        return;
    }
    lastClickTime = now;

    const targetUrl = getSmartVideoUrl(anchor);
    if (!targetUrl) return showToast("未能获取链接");

    const taskKey = targetUrl + type;
    if (processingSet.has(taskKey)) {
        showToast("任务进行中，请查看侧边栏");
        return;
    }

    toggleLoading(mainBtn, true);

    // 从按钮附近提取作者和标题
    const { author, title } = extractAuthorAndTitleFromAnchor(anchor);

    try {
        chrome.runtime.sendMessage({
            action: "checkLimit",
            type: type
        }, (res) => {
            if (chrome.runtime.lastError) {
                toggleLoading(mainBtn, false);
                showToast("插件已更新，请 F5 刷新网页");
                return;
            }

            // 🔥 检查是否登录，未登录则醒目提示
            if (res && !res.isLoggedIn) {
                toggleLoading(mainBtn, false);
                showLoginToast();
                return;
            }

            processingSet.add(taskKey);

            let actionName = "";
            let toastMsg = "";

            // 🔥 文案提取：允许免费用户使用（后端会检查额度）
            if (type === "copy") {
                actionName = "transcribeAudio";

                // 🔥 实时从服务器获取用户状态，避免使用过期的缓存
                chrome.storage.local.get(['token', 'user_info'], async (storageRes) => {
                    let isVip = false;

                    if (storageRes.token) {
                        try {
                            const res = await fetch('https://zhushou.xokx.top/api/user/me', {
                                headers: { 'Authorization': `Bearer ${storageRes.token}` }
                            });
                            if (res.ok) {
                                const data = await res.json();
                                if (data.user && data.user.isVip) {
                                    isVip = true;
                                }
                            }
                        } catch (e) {
                            // 静默处理，不输出日志
                        }
                    }

                    // VIP 用户显示"已加入队列"，免费用户显示"正在处理中..."
                    toastMsg = isVip ? "已加入后台队列" : "正在处理中...";
                    showToast(toastMsg);
                    chrome.runtime.sendMessage({ action: "openSidePanel" });
                    sendTaskMessage(actionName, type, targetUrl, author, title, mainBtn, taskKey);
                });
                return;
            }

            if (type === "video") {
                actionName = "downloadMedia";
                toastMsg = "开始下载";
            } else if (type === "audio") {
                actionName = "downloadMedia";
                toastMsg = "开始下载";
            }

            showToast(toastMsg);

            sendTaskMessage(actionName, type, targetUrl, author, title, mainBtn, taskKey);
        });
    } catch (err) {
        toggleLoading(mainBtn, false);
        showToast("插件已更新，请 F5 刷新网页");
    }

    // 🔥 发送任务消息的函数
    function sendTaskMessage(actionName, type, targetUrl, author, title, mainBtn, taskKey) {
        chrome.runtime.sendMessage({
            action: actionName,
            type: type,
            videoUrl: targetUrl,
            pageAuthor: author,
            pageTitle: title
        }, (response) => {
            if (chrome.runtime.lastError) {
                toggleLoading(mainBtn, false);
                processingSet.delete(taskKey);
                return;
            }

            if (response && (response.status === "success" || response.status === "started" || response.status === "queued")) {
                if (type === "copy") {
                    const checkIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#10b981" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg>`;
                    showToast("已加入队列，请查看侧边栏", checkIcon);
                    setTimeout(() => {
                        toggleLoading(mainBtn, false);
                        setTimeout(() => processingSet.delete(taskKey), 10000);
                    }, 2000);
                } else {
                    toggleLoading(mainBtn, false);
                    processingSet.delete(taskKey);
                }
            } else {
                toggleLoading(mainBtn, false);
                processingSet.delete(taskKey);
                const errorIcon = `<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#ef4444" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>`;
                showToast("失败: " + (response?.msg || "未知错误"), errorIcon);
            }
        });
    }

    setTimeout(() => {
        if (mainBtn.classList.contains('loading')) {
            toggleLoading(mainBtn, false);
            processingSet.delete(taskKey);
        }
    }, 60000);
}

// 清理作者名：移除认证徽章、emoji和合集相关文本
function cleanAuthorName(text) {
    if (!text) return text;
    // 移除常见的认证徽章符号和emoji
    text = text.replace(/^[\u2605\u2665\u2764\uf005\u2B50\u2600\u26A1]+/g, '');
    text = text.replace(/^[✓✔✅]+/g, '');
    text = text.replace(/认证|已认证|AUTHENTICATED/gi, '');
    // 移除合集相关文本
    text = text.replace(/合集|系列|收藏|喜欢|playlist/gi, '');
    // 移除括号及其内容（如：（载十倍音质♫））
    text = text.replace(/（[^）]*）/g, '');
    text = text.replace(/\([^)]*\)/g, '');
    // 移除音乐符号等特殊字符
    text = text.replace(/[♪♫♬♩♭♮♯]/g, '');
    // 移除开头的特殊符号（保留中文、英文、数字、下划线）
    text = text.replace(/^[^\u4e00-\u9fa5a-zA-Z0-9_]+/g, '');
    return text.trim();
}

function extractAuthorAndTitleFromAnchor(anchor) {
    let author = "Douyin";
    let title = "Video";

    try {
        // 🔥 根据 URL 判断页面类型，而不是 DOM 结构
        const isSearchPage = window.location.href.includes('/search/');
        const isRecommendPage = window.location.pathname === '/' || window.location.pathname.includes('/feed');

        let videoCard = anchor.closest('[data-e2e-vid]');
        if (!videoCard) videoCard = anchor.closest('.xg-video-container')?.parentElement;
        if (!videoCard) videoCard = anchor.closest('[class*="video-item"]') || anchor.closest('[class*="feed-item"]');
        if (!videoCard) videoCard = anchor.closest('a[href*="/video/"]');
        if (!videoCard) videoCard = anchor.closest('.xg-right-grid') || anchor.parentNode?.parentNode;

        if (!videoCard) {
            return { author, title };
        }

        // ===== 搜索页专用逻辑 =====
        if (isSearchPage) {
            // 搜索页：查找 search-result-card 并提取作者
            let searchCard = null;
            let parent = videoCard;
            for (let i = 0; i < 10; i++) {
                parent = parent?.parentElement;
                if (!parent || parent === document.body) break;
                if (parent.classList.contains('search-result-card')) {
                    searchCard = parent;
                    break;
                }
            }

            if (searchCard) {
                const userLinks = searchCard.querySelectorAll('a[href*="/user/"]');
                for (const link of userLinks) {
                    let text = link.innerText?.trim();
                    if (!text || text === '认证徽章' || text === '蓝V' || text === '金V' || text.includes('认证')) {
                        continue;
                    }
                    text = text?.split('\n')[0]?.trim();
                    text = text?.replace(/·.*$/gm, '').trim();
                    text = cleanAuthorName(text);
                    if (text && text.length > 1 && text.length < 30) {
                        author = text;
                        break;
                    }
                }
            }

            // 搜索页标题提取
            if (title === "Video") {
                const allEls = searchCard?.querySelectorAll('*') || videoCard.querySelectorAll('*');
                for (const el of allEls) {
                    const t = el.innerText?.trim();
                    if (!t || t.includes('相关搜索') || t.includes('推荐') ||
                        t.includes('合集') || t.includes('系列') ||
                        t.includes('collection') || t.includes('series') ||
                        t.includes('playlist') || t.includes('http')) continue;

                    if (t.length > 10 && t.length < 200) {
                        let cleanTitle = t.replace(/\n+/g, ' ').replace(/\s+/g, ' ').trim();
                        if (cleanTitle.includes('认证徽章') || cleanTitle.includes('蓝V') || cleanTitle.includes('金V')) {
                            continue;
                        }
                        cleanTitle = cleanTitle.replace(/^第\d+集[：:]\s*/, '');
                        cleanTitle = cleanTitle.replace(/^【合集】/, '');
                        cleanTitle = cleanTitle.replace(/·\d+[月年]前/g, '');
                        cleanTitle = cleanTitle.replace(/·\d+天前/g, '');
                        cleanTitle = cleanTitle.replace(/·刚刚/g, '');
                        cleanTitle = cleanTitle.replace(/·[\u4e00-\u9fa5]+前/g, '');
                        cleanTitle = cleanTitle.replace(/（[^）]*）/g, '');
                        cleanTitle = cleanTitle.replace(/\([^)]*\)/g, '');
                        if (cleanTitle.startsWith(author)) {
                            cleanTitle = cleanTitle.substring(author.length).trim();
                        }
                        if (cleanTitle === author || cleanTitle.replace(/\s+/g, '') === author) {
                            continue;
                        }
                        cleanTitle = cleanTitle.replace(/^[\s·]+/, '').trim();
                        let candidateTitle = cleanTitle.substring(0, 50);
                        if (candidateTitle && candidateTitle !== author && candidateTitle.length > 8) {
                            title = candidateTitle;
                            break;
                        }
                    }
                }
            }
        }
        // ===== 推荐页（主页）专用逻辑 =====
        else {
            // 方法1: 查找 @ 格式作者名
            let found = false;
            const atPattern = /^@[\u4e00-\u9fa5a-zA-Z0-9_]{2,20}$/;
            const candidates = videoCard.querySelectorAll('[data-e2e="video-author-name"], [data-e2e="browse-user-name"], [data-e2e="feed-author"], [data-e2e="recommend-author"], a[href*="/user/"], [class*="author"], [class*="nickname"], [class*="user-name"], [class*="username"]');

            for (const el of candidates) {
                if (el.tagName === 'svg' || el.tagName === 'path' || el.tagName === 'use') continue;
                let text = el.innerText?.trim();
                text = cleanAuthorName(text);
                if (text && atPattern.test(text)) {
                    author = text;
                    found = true;
                    break;
                }
            }

            if (!found) {
                const authorSelectors = [
                    '[data-e2e="feed-video-nickname"]',  // 推荐页专用
                    '[data-e2e="video-author-name"]',
                    '[data-e2e="browse-user-name"]',
                    '[data-e2e="feed-author"]',
                    '[data-e2e="recommend-author"]',
                    'a[href*="/user/"]',
                    '[class*="author"]',
                    '[class*="nickname"]',
                    '[class*="user-name"]',
                    '[class*="username"]'
                ];

                for (const sel of authorSelectors) {
                    const el = videoCard.querySelector(sel);
                    if (el && (el.tagName === 'svg' || el.tagName === 'path' || el.tagName === 'use')) continue;
                    if (el && el.innerText) {
                        let text = el.innerText.trim();
                        text = cleanAuthorName(text);

                        // 🔥 尝试提取 .arnSiSbK 内部文本（推荐页作者名嵌套在这里）
                        const innerAuthor = el.querySelector('.arnSiSbK');
                        if (innerAuthor) {
                            text = innerAuthor.innerText?.trim() || text;
                            text = cleanAuthorName(text);
                        }
                        // 如果是链接元素，额外处理
                        else if (el.tagName === 'A' && el.href.includes('/user/')) {
                            text = el.innerText?.trim().split('\n')[0]?.trim() || text;
                            text = cleanAuthorName(text);
                        }

                        if (text && text.length > 0 && text.length < 30 && !text.includes('\n')) {
                            author = text;
                            break;
                        }
                    }
                }

                // 备用方案：如果选择器都找不到作者，查找所有用户链接
                if (author === "Douyin") {
                    const allLinks = videoCard.querySelectorAll('a[href*="/user/"]');
                    for (const link of allLinks) {
                        let text = link.innerText?.trim();
                        if (!text || text === '认证徽章' || text === '蓝V' || text === '金V' || text.includes('认证')) {
                            continue;
                        }
                        text = text?.split('\n')[0]?.trim();
                        text = cleanAuthorName(text);
                        if (text && text.length > 1 && text.length < 30) {
                            author = text;
                            break;
                        }
                    }
                }

                // 终极备用方案：遍历所有元素，查找可能是作者名的短文本
                if (author === "Douyin") {
                    const candidates = [];
                    const allElements = videoCard.querySelectorAll('*');
                    for (const el of allElements) {
                        if (el.tagName === 'BUTTON' || el.tagName === 'svg' ||
                            el.tagName === 'path' || el.tagName === 'use' ||
                            el.tagName === 'IMG' || el.tagName === 'VIDEO' ||
                            el.tagName === 'SCRIPT' || el.tagName === 'STYLE') {
                            continue;
                        }
                        const t = el.innerText?.trim();
                        if (!t || t.length < 2 || t.length > 25) continue;
                        // 跳过明显不是作者名的文本
                        if (t.includes('点赞') || t.includes('评论') || t.includes('分享') ||
                            t.includes('收藏') || t.includes('转发') || t.includes('关注') ||
                            t.includes('合集') || t.includes('系列') || t.includes('推荐')) {
                            continue;
                        }
                        // 跳过包含数字太多的文本（可能是标题）
                        if ((t.match(/\d/g) || []).length > 5) continue;
                        // 跳过包含特殊符号太多的文本
                        if ((t.match(/[#￥$%^&*]/g) || []).length > 2) continue;
                        candidates.push({ text: t, element: el });
                    }
                    // 按长度排序，选择最短且包含 @ 或中文的文本
                    candidates.sort((a, b) => a.text.length - b.text.length);
                    for (const c of candidates) {
                        const cleaned = cleanAuthorName(c.text);
                        if (cleaned && cleaned.length >= 2 && cleaned.length < 20) {
                            // 优先选择包含 @ 符号的文本
                            if (cleaned.includes('@') || /^[\u4e00-\u9fa5a-zA-Z0-9_]{2,15}$/.test(cleaned)) {
                                author = cleaned;
                                break;
                            }
                        }
                    }
                    // 如果没找到 @ 格式的，选择最短的
                    if (author === "Douyin" && candidates.length > 0) {
                        const cleaned = cleanAuthorName(candidates[0].text);
                        if (cleaned && cleaned.length >= 2) {
                            author = cleaned;
                        }
                    }
                }
            }

            // ===== 推荐页标题提取 =====
            if (title === "Video") {
                const titleSelectors = [
                    '[data-e2e="video-desc"]',
                    '[data-e2e="browse-video-desc"]',
                    '[data-e2e="feed-desc"]',
                    '[data-e2e="recommend-desc"]',
                    '[class*="desc"]',
                    '[class*="title"]',
                    '[class*="content"]',
                    '[class*="text"]'
                ];

                for (const sel of titleSelectors) {
                    const el = videoCard.querySelector(sel);
                    if (el && el.innerText && el.innerText.trim()) {
                        let t = el.innerText.trim();
                        t = t.replace(/^第\d+集[：:]\s*/, '');
                        t = t.replace(/^【合集】/, '');
                        if (t && t.length > 0 && t.length < 100) {
                            title = t.substring(0, 50);
                            break;
                        }
                    }
                }

                // 备用方案：遍历所有元素查找最长的文本
                if (title === "Video") {
                    const allTexts = [];
                    const allElements = videoCard.querySelectorAll('*');
                    for (const el of allElements) {
                        if (el.tagName === 'BUTTON' || el.tagName === 'svg' ||
                            el.tagName === 'path' || el.tagName === 'use' ||
                            el.tagName === 'IMG' || el.tagName === 'VIDEO') {
                            continue;
                        }
                        const t = el.innerText?.trim();
                        if (!t || t.length < 5 || t.length > 200) continue;

                        // 🔥 过滤按钮文字和操作提示
                        const ignoreKeywords = [
                            '点赞', '评论', '分享', '收藏', '转发', '关注',
                            '文案提取', '音频下载', '视频下载',
                            '抢首评', '智能', '倍速', '大家都'
                        ];
                        if (ignoreKeywords.some(kw => t.includes(kw))) {
                            continue;
                        }

                        // 🔥 跳过包含多个空格分隔的短词（可能是按钮组文字）
                        const words = t.split(/\s+/);
                        if (words.length > 5 && words.every(w => w.length < 5)) {
                            continue;
                        }

                        allTexts.push({ text: t, element: el });
                    }
                    if (allTexts.length > 0) {
                        allTexts.sort((a, b) => b.text.length - a.text.length);
                        let bestTitle = allTexts[0].text;
                        bestTitle = bestTitle.replace(/^第\d+集[：:]\s*/, '');
                        bestTitle = bestTitle.replace(/^【合集】/, '');
                        if (bestTitle && bestTitle.length > 5) {
                            title = bestTitle.substring(0, 50);
                        }
                    }
                }
            }
        }
    } catch (e) {
        // 静默处理错误
    }

    return { author, title };
}

function getSmartVideoUrl(anchor) {
    // 方法1: 从 data-e2e-vid 属性获取
    let container = anchor.closest('[data-e2e-vid]');
    if (container) {
        const vid = container.getAttribute('data-e2e-vid');
        if (vid) {
            return `https://www.douyin.com/video/${vid}`;
        }
    }

    // 方法2: 从视频播放器容器获取
    container = anchor.closest('.xg-video-container');
    if (!container) container = anchor.closest('[data-e2e="feed-active-video"]');
    if (!container) container = anchor.closest('[data-e2e="browse-video-desc"]')?.closest('[class*="video"]');

    if (container) {
        const vid = container.getAttribute('data-e2e-vid') ||
                    container.getAttribute('data-video-id') ||
                    container.getAttribute('id')?.match(/video-(\d+)/)?.[1];

        if (vid) {
            return `https://www.douyin.com/video/${vid}`;
        }
    }

    // 方法3: 从搜索结果页面的视频卡片获取
    const videoCardLink = anchor.closest('a[href*="/video/"]');
    if (videoCardLink) {
        const href = videoCardLink.getAttribute('href');
        if (href && href.includes('/video/')) {
            return href.startsWith('http') ? href : 'https://www.douyin.com' + href;
        }
    }

    // 方法4: 查找最近的视频链接
    const allVideoLinks = document.querySelectorAll('a[href*="/video/"]');
    for (const link of allVideoLinks) {
        const rect = anchor.getBoundingClientRect();
        const linkRect = link.getBoundingClientRect();
        const distance = Math.abs(rect.top - linkRect.top) + Math.abs(rect.left - linkRect.left);

        if (distance < 200) {
            const href = link.getAttribute('href');
            if (href && href.includes('/video/')) {
                return href.startsWith('http') ? href : 'https://www.douyin.com' + href;
            }
        }
    }

    // 方法5: 当前页面 URL
    if (window.location.href.includes('/video/')) {
        return window.location.href;
    }

    return null;
}

function showToast(msg, icon = null) {
    const old = document.querySelector('.dy-copy-toast');
    if (old) old.remove();
    const t = document.createElement('div');
    t.className = 'dy-copy-toast';

    // 支持传入 SVG 图标
    if (icon) {
        t.innerHTML = `
            <div style="display:flex;align-items:center;gap:8px;">
                ${icon}
                <span>${msg}</span>
            </div>
        `;
    } else {
        t.innerText = msg;
    }

    document.body.appendChild(t);

    // 3.5秒后添加隐藏动画，然后移除
    setTimeout(() => {
        t.classList.add('hiding');
        setTimeout(() => t.remove(), 300);
    }, 3500);
}

// 🔥 醒目的登录提示
function showLoginToast() {
    const old = document.querySelector('.dy-login-toast');
    if (old) old.remove();
    const t = document.createElement('div');
    t.className = 'dy-login-toast';
    t.innerHTML = `
        <div style="display:flex;align-items:center;gap:12px;justify-content:center;">
            <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <rect x="3" y="11" width="18" height="11" rx="2" ry="2"></rect>
                <path d="M7 11V7a5 5 0 0 1 9.9-1"></path>
            </svg>
            <div style="text-align:left;">
                <div style="font-size:18px;font-weight:bold;margin-bottom:6px;">请先登录</div>
                <div style="font-size:14px;font-weight:normal;opacity:0.9;">请在右侧面板完成登录</div>
            </div>
        </div>
    `;
    document.body.appendChild(t);
    setTimeout(() => t.remove(), 5000);
}

// 🔥 醒目的 VIP 提示
function showVipToast() {
    const old = document.querySelector('.dy-login-toast');
    if (old) old.remove();
    const t = document.createElement('div');
    t.className = 'dy-login-toast';
    t.innerHTML = `
        <div style="display:flex;align-items:center;gap:12px;justify-content:center;">
            <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"></polygon>
            </svg>
            <div style="text-align:left;">
                <div style="font-size:18px;font-weight:bold;margin-bottom:6px;">VIP 专属功能</div>
                <div style="font-size:14px;font-weight:normal;opacity:0.9;">文案提取需要升级 VIP</div>
            </div>
        </div>
    `;
    document.body.appendChild(t);
    setTimeout(() => t.remove(), 6000);
}

// 🔥 醒目的额度超限提示
function showLimitExceededToast(message, upgradeUrl) {
    // 移除旧的提示（包括登录提示和普通 copy 提示）
    const oldLogin = document.querySelector('.dy-login-toast');
    if (oldLogin) oldLogin.remove();
    const oldCopy = document.querySelector('.dy-copy-toast');
    if (oldCopy) oldCopy.remove();

    const t = document.createElement('div');
    t.className = 'dy-login-toast';

    // 将多行消息转换为 HTML
    const messageHtml = message.replace(/\n/g, '<br>').replace(/📊|🎁/g, '');

    // 根据错误消息判断类型
    let title = "免费额度已用完";
    let showButton = false;
    if (message.includes('系统文案提取正在升级')) {
        title = "系统升级中";
    } else if (message.includes('积分') || upgradeUrl?.includes('recharge')) {
        title = "积分不足";
        showButton = true;
    }

    // 🔥 添加充值按钮（如果是积分相关提示）
    const buttonHtml = showButton && upgradeUrl ? `
        <button id="dy-recharge-btn" data-url="${upgradeUrl}"
            style="margin-top:10px;padding:8px 20px;background:#d97706;color:#fff;border:none;border-radius:8px;cursor:pointer;font-size:14px;font-weight:600;transition:all 0.2s;"
            onmouseover="this.style.background='#b45309'"
            onmouseout="this.style.background='#d97706'">
            立即充值
        </button>
    ` : '';

    // 如果有充值按钮，添加点击事件
    if (showButton && upgradeUrl) {
        setTimeout(() => {
            const btn = document.getElementById('dy-recharge-btn');
            if (btn) {
                btn.addEventListener('click', async (e) => {
                    e.preventDefault();
                    chrome.storage.local.get(['token'], async (res) => {
                        if (res.token) {
                            // 请求临时票据
                            try {
                                const response = await fetch('https://zhushou.xokx.top/api/auth/ticket', {
                                    method: 'POST',
                                    headers: {
                                        'Content-Type': 'application/json',
                                        'Authorization': `Bearer ${res.token}`
                                    },
                                    body: JSON.stringify({ type: 'recharge' })
                                });

                                if (response.ok) {
                                    const data = await response.json();
                                    // 使用ticket打开充值页面
                                    const url = `https://zhushou.xokx.top${upgradeUrl}#ticket=${data.ticket}`;
                                    window.open(url, '_blank');
                                } else {
                                    alert('获取票据失败，请重试');
                                }
                            } catch (e) {
                                console.error('[获取票据] 错误:', e);
                                alert('网络错误，请重试');
                            }
                        } else {
                            // 没有token，直接打开页面（会提示登录）
                            window.open(`https://zhushou.xokx.top${upgradeUrl}`, '_blank');
                        }
                    });
                });
            }
        }, 100);
    }

    t.innerHTML = `
        <div style="display:flex;align-items:center;gap:12px;justify-content:center;">
            <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="#f59e0b" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <circle cx="12" cy="12" r="10"/>
                <line x1="12" y1="8" x2="12" y2="12"/>
                <line x1="12" y1="16" x2="12.01" y2="16"/>
            </svg>
            <div style="text-align:left;">
                <div style="font-size:18px;font-weight:bold;margin-bottom:6px;">${title}</div>
                <div style="font-size:14px;font-weight:normal;opacity:0.9;line-height:1.6;">${messageHtml}</div>
                ${buttonHtml}
            </div>
        </div>
    `;

    document.body.appendChild(t);
    setTimeout(() => t.remove(), 8000);
}

// 🔥 监听来自 background.js 的消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === "showLimitExceededToast") {
        showLimitExceededToast(request.message, request.upgradeUrl || "/pages/pay.html");
        sendResponse({ received: true });
    }
    // 🔥 全屏分屏弹窗
    if (request.action === "showSplitModal") {
        showSplitModal(request.data);
        sendResponse({ received: true });
    }
    if (request.action === "closeSplitModal") {
        closeSplitModal();
        sendResponse({ received: true });
    }
    if (request.action === "updateSplitModalContent") {
        updateSplitModalContent(request.data);
        sendResponse({ received: true });
    }
    // 保存响应
    if (request.action === "saveCloudRecordResponse") {
        handleSaveResponse(request);
        sendResponse({ received: true });
    }
    return true; // 保持消息通道开放
});

// 处理保存响应
let pendingSaveCallback = null;
function handleSaveResponse(response) {
    if (pendingSaveCallback) {
        pendingSaveCallback(response);
        pendingSaveCallback = null;
    }
}

// ==================== 全屏分屏弹窗 ====================
let splitModalOverlay = null;
let splitOriginalTextarea = null;
let splitRewriteTextarea = null;
let currentRecordId = null;

function showSplitModal(data) {
    // 如果已存在，先关闭
    closeSplitModal();

    currentRecordId = data._id;

    // 创建全屏遮罩层
    splitModalOverlay = document.createElement('div');
    splitModalOverlay.id = 'dy-split-modal-overlay';
    splitModalOverlay.innerHTML = `
        <div class="dy-split-modal">
            <div class="dy-split-header">
                <h3>原文 / 二创对比</h3>
                <button class="dy-split-close">×</button>
            </div>
            <div class="dy-split-content">
                <div class="dy-split-panel">
                    <div class="dy-split-panel-header">
                        <span>原文</span>
                        <button class="dy-split-copy" data-type="original">复制</button>
                    </div>
                    <textarea class="dy-split-textarea" placeholder="原文内容">${(data.originalText || '').replace(/</g, '&lt;')}</textarea>
                </div>
                <div class="dy-split-panel">
                    <div class="dy-split-panel-header">
                        <span>二创</span>
                        <button class="dy-split-copy" data-type="rewrite">复制</button>
                    </div>
                    <textarea class="dy-split-textarea" placeholder="二创文案">${(data.rewriteText || '').replace(/</g, '&lt;')}</textarea>
                </div>
            </div>
            <div class="dy-split-footer">
                <button class="dy-split-btn-cancel">取消</button>
                <button class="dy-split-btn-save">保存修改</button>
            </div>
        </div>
    `;

    document.body.appendChild(splitModalOverlay);

    // 获取元素引用
    splitOriginalTextarea = splitModalOverlay.querySelector('.dy-split-panel:first-child .dy-split-textarea');
    splitRewriteTextarea = splitModalOverlay.querySelector('.dy-split-panel:last-child .dy-split-textarea');

    // 绑定事件
    const closeBtn = splitModalOverlay.querySelector('.dy-split-close');
    const cancelBtn = splitModalOverlay.querySelector('.dy-split-btn-cancel');
    const saveBtn = splitModalOverlay.querySelector('.dy-split-btn-save');
    const copyBtns = splitModalOverlay.querySelectorAll('.dy-split-copy');

    closeBtn.onclick = closeSplitModal;
    cancelBtn.onclick = closeSplitModal;
    splitModalOverlay.onclick = (e) => {
        if (e.target === splitModalOverlay) closeSplitModal();
    };

    // 复制按钮
    copyBtns.forEach(btn => {
        btn.onclick = (e) => {
            const type = e.target.dataset.type;
            const textarea = type === 'original' ? splitOriginalTextarea : splitRewriteTextarea;
            navigator.clipboard.writeText(textarea.value).then(() => {
                showToast(type === 'original' ? '原文已复制' : '二创已复制');
            });
        };
    });

    // 保存按钮
    saveBtn.onclick = async () => {
        const newOriginalText = splitOriginalTextarea.value.trim();
        const newRewriteText = splitRewriteTextarea.value.trim();

        if (!newRewriteText) {
            showToast('二创内容不能为空');
            return;
        }

        // 发送保存请求到 panel
        chrome.runtime.sendMessage({
            action: 'saveCloudRecord',
            recordId: currentRecordId,
            originalText: newOriginalText,
            rewriteText: newRewriteText
        }, (response) => {
            if (response && response.success) {
                showToast('保存成功');
                closeSplitModal();
                // 通知 panel 刷新列表
                chrome.runtime.sendMessage({ action: 'refreshCloudRecords' });
            } else {
                showToast('保存失败：' + (response?.error || '未知错误'));
            }
        });
    };
}

function closeSplitModal() {
    if (splitModalOverlay) {
        splitModalOverlay.remove();
        splitModalOverlay = null;
        splitOriginalTextarea = null;
        splitRewriteTextarea = null;
        currentRecordId = null;
    }
}

function updateSplitModalContent(data) {
    if (splitOriginalTextarea && splitRewriteTextarea) {
        splitOriginalTextarea.value = data.originalText || '';
        splitRewriteTextarea.value = data.rewriteText || '';
        currentRecordId = data._id;
    }
}
