// extraction-modal.js - 提取记录编辑窗口逻辑

let currentExtractionIndex = null;
let currentExtractionData = null;

// 获取 DOM 元素
const modalOverlay = document.getElementById('modalOverlay');
const extractionText = document.getElementById('extractionText');
const closeBtn = document.getElementById('closeBtn');
const cancelBtn = document.getElementById('cancelBtn');
const saveBtn = document.getElementById('saveBtn');
const copyBtn = document.getElementById('copyBtn');
const charCount = document.getElementById('charCount');

// 更新字数统计
function updateCharCount() {
    charCount.textContent = `${extractionText.value.length}字`;
}

// 显示 Toast 提示
function showToast(message) {
    const oldToast = document.querySelector('.dy-copy-toast');
    if (oldToast) oldToast.remove();

    const toast = document.createElement('div');
    toast.className = 'dy-copy-toast';
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => {
        toast.classList.add('hiding');
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

// 初始化弹窗数据
function initModal(data) {
    currentExtractionIndex = data.index;
    currentExtractionData = data;
    extractionText.value = data.content || '';
    updateCharCount();
}

// 关闭窗口
function closeModal() {
    window.close();
}

// 绑定事件
closeBtn.addEventListener('click', closeModal);
cancelBtn.addEventListener('click', closeModal);

modalOverlay.addEventListener('click', (e) => {
    if (e.target === modalOverlay) closeModal();
});

// 监听输入事件，实时更新字数
extractionText.addEventListener('input', updateCharCount);

// 复制按钮
copyBtn.addEventListener('click', () => {
    navigator.clipboard.writeText(extractionText.value).then(() => {
        showToast('已复制');
    });
});

// 保存按钮
saveBtn.addEventListener('click', async () => {
    const newContent = extractionText.value.trim();

    if (!newContent) {
        showToast('内容不能为空');
        return;
    }

    // 发送保存请求到 background.js
    chrome.runtime.sendMessage({
        action: 'saveExtractionRecord',
        index: currentExtractionIndex,
        content: newContent
    }, (response) => {
        if (chrome.runtime.lastError) {
            showToast('保存失败：' + chrome.runtime.lastError.message);
            return;
        }

        if (response && response.success) {
            showToast('保存成功');
            setTimeout(() => {
                closeModal();
            }, 500);
        } else {
            showToast('保存失败：' + (response?.error || '未知错误'));
        }
    });
});

// 监听来自 panel 的初始化消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'initExtractionModal') {
        initModal(request.data);
        sendResponse({ received: true });
    }
    return true;
});

// 窗口加载完成后通知 background
window.addEventListener('load', () => {
    chrome.runtime.sendMessage({
        action: 'extractionModalWindowReady'
    });
});

// 窗口关闭前清理
window.addEventListener('beforeunload', () => {
    chrome.runtime.sendMessage({
        action: 'extractionModalWindowClosed'
    });
});
