// panel.js - v4.85 三个功能按钮 iOS 毛玻璃风格
console.log('[panel.js] 文件已加载,版本: v4.85');

const historyList = document.getElementById('history-list');
const activeTaskList = document.getElementById('active-task-list');
const statusText = document.getElementById('status-text');
const clearBtn = document.getElementById('clear-all');
const openFolderBtn = document.getElementById('open-folder');
const modelSelect = document.getElementById('ai-model-select');

// 搜索相关 DOM
const searchInput = document.getElementById('search-input');
const clearSearchBtn = document.getElementById('clear-search-btn');

// 存储所有提取记录（用于搜索过滤）
let allHistoryItems = [];

// Auth DOM
const notLoggedInDiv = document.getElementById('not-logged-in');
const loggedInDiv = document.getElementById('logged-in');
const avatarImg = document.getElementById('avatar-img');
const authModal = document.getElementById('auth-modal');
const profileModal = document.getElementById('profile-modal');
const closeAuthBtn = document.getElementById('close-auth');
const closeProfileBtn = document.getElementById('close-profile');
const authTabs = document.querySelectorAll('.auth-tab');
const tabLogin = document.getElementById('tab-login');
const tabRegister = document.getElementById('tab-register');
const loginUsername = document.getElementById('login-username');
const loginPassword = document.getElementById('login-password');
const btnDoLogin = document.getElementById('btn-do-login');
const loginMsg = document.getElementById('login-msg');

// 注册相关 DOM
const regUsername = document.getElementById('reg-username');
const regPassword = document.getElementById('reg-password');
const regPhone = document.getElementById('reg-phone'); // 新增
const regCode = document.getElementById('reg-code');   // 新增
const btnSendCode = document.getElementById('btn-send-code'); // 新增
const btnDoRegister = document.getElementById('btn-do-register');
const regMsg = document.getElementById('reg-msg');
const errorUsername = document.getElementById('error-username');
const errorPassword = document.getElementById('error-password');

const profileAvatarImg = document.getElementById('profile-avatar-img');
const displayName = document.getElementById('display-name');
const logoutBtn = document.getElementById('logout-btn');

// 菜单 DOM
const menuUsername = document.getElementById('menu-username');
const menuRole = document.getElementById('menu-role');
const vipExpireDateEl = document.getElementById('vip-expire-date');
const profileExpireText = document.getElementById('profile-expire-text');
const profileBadge = document.getElementById('profile-badge');
const goVipBtn = document.getElementById('go-vip-btn');
const goProfileBtn = document.getElementById('go-profile-btn');
const menuLogoutBtn = document.getElementById('menu-logout-btn');

// 顶部链接 DOM
const linkManual = document.getElementById('link-manual');
const linkPrivacy = document.getElementById('link-privacy');

// 编辑器 DOM
const editModal = document.getElementById('edit-modal');
const largeEditorArea = document.getElementById('large-editor-area');
const btnEditCancel = document.getElementById('btn-edit-cancel');
const btnEditSave = document.getElementById('btn-edit-save');
const closeEditBtn = document.getElementById('close-edit');

// AI 二创 DOM
const rewriteResultArea = document.getElementById('rewrite-result-area');
const rewriteContent = document.getElementById('rewrite-content');
const aiHeader = document.getElementById('ai-header');
const btnCopyRewrite = document.getElementById('btn-copy-rewrite');
const btnSaveRewriteCloud = document.getElementById('btn-save-rewrite-cloud');

// 云端记录弹窗 DOM
const cloudModal = document.getElementById('cloud-modal');
const closeCloudModalBtn = document.getElementById('close-cloud-modal');
const openCloudModalBtn = document.getElementById('btn-open-cloud-modal');
const cloudModalList = document.getElementById('cloud-modal-list');

// 分屏弹窗 DOM
const splitModal = document.getElementById('split-modal');
const closeSplitModalBtn = document.getElementById('close-split-modal');
const splitOriginalText = document.getElementById('split-original-text');
const splitRewriteText = document.getElementById('split-rewrite-text');
const splitSaveBtn = document.getElementById('split-save-btn');
const splitCancelBtn = document.getElementById('split-cancel-btn');
const splitCopyOriginal = document.getElementById('split-copy-original');
const splitCopyRewrite = document.getElementById('split-copy-rewrite');

// 确认弹窗 DOM
const confirmModal = document.getElementById('confirm-modal');
const confirmDesc = document.getElementById('confirm-desc');
const btnConfirmCancel = document.getElementById('btn-confirm-cancel');
const btnConfirmOk = document.getElementById('btn-confirm-ok');

let currentEditIndex = -1;
let currentItems = [];
let isEditingRewrite = false;
let currentRewriteOriginalText = ''; // 当前二创的原文

// 云端记录状态
let currentCloudRecords = [];
let currentEditCloudIndex = -1;
let currentSplitRecordId = null; // 当前编辑的分屏记录

// 提取记录弹窗窗口数据
let extractionModalWindowData = null;

// 二创结果弹窗窗口数据
let rewriteModalWindowData = null;

// 🔥 删除警告状态索引（用于提取记录删除确认）
let deleteWarningIndex = -1;

const API_BASE = "https://zhushou.xokx.top/api";

// 分段功能已移至后端，前端不再处理

const usernameRegex = /^[a-zA-Z0-9]+$/;

function fixModalPosition() {
    const modals = [authModal, profileModal, editModal, confirmModal, cloudModal, splitModal, document.getElementById('global-toast'), document.getElementById('notice-modal')];
    modals.forEach(m => { if (m && m.parentNode !== document.body) document.body.appendChild(m); });
}
fixModalPosition();

function showToast(msg, type = 'default') {
    const toast = document.getElementById('global-toast');
    if (!toast) return;
    toast.className = 'global-toast';
    if (type !== 'default') toast.classList.add(type);
    toast.innerText = msg;
    toast.classList.add('show');
    if (toast.timeout) clearTimeout(toast.timeout);
    toast.timeout = setTimeout(() => { toast.classList.remove('show'); }, 2000);
}

function showConfirm(text, onConfirm) {
    if (!confirmModal) return;
    confirmDesc.innerText = text;
    confirmModal.classList.remove('hidden');
    btnConfirmCancel.onclick = () => { confirmModal.classList.add('hidden'); };
    btnConfirmOk.onclick = () => { confirmModal.classList.add('hidden'); if (onConfirm) onConfirm(); };
}

function showAuthMsg(el, txt, type) {
    if (!el) return;
    el.innerText = txt;
    el.className = "auth-msg";
    if (type === "error") el.classList.add("error");
    if (type === "success") el.classList.add("success");
}

function validateInput(input, errorEl, ruleFn, errorMsg) {
    if (!input) return false;
    const wrapper = input.parentElement;
    const validIcon = wrapper.querySelector('.valid-icon');
    const val = input.value.trim();

    input.classList.remove('input-valid', 'input-invalid');
    if(errorEl) errorEl.innerText = "";
    if(validIcon) validIcon.classList.add('hidden');

    if (!val) return false;

    if (ruleFn(val)) {
        input.classList.add('input-valid');
        if(validIcon) validIcon.classList.remove('hidden');
        return true;
    } else {
        input.classList.add('input-invalid');
        if(errorEl) errorEl.innerText = errorMsg;
        return false;
    }
}

// 密码眼睛切换
document.querySelectorAll('.toggle-pwd').forEach(btn => {
    btn.addEventListener('click', (e) => {
        const input = btn.parentElement.querySelector('input');
        if(!input) return;
        if (input.type === 'password') {
            input.type = 'text';
            btn.innerHTML = `<svg class="icon-eye" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>`;
        } else {
            input.type = 'password';
            btn.innerHTML = `<svg class="icon-eye-off" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"></path><line x1="1" y1="1" x2="23" y2="23"></line></svg>`;
        }
    });
});

// === 1. 初始化 ===

// 🔥 监听登录模态框打开标志位
chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'local' && changes._openLoginModal && changes._openLoginModal.newValue === true) {
        // 清除标志位
        chrome.storage.local.remove('_openLoginModal');
        // 打开登录框
        if (authModal) {
            authModal.classList.remove('hidden');
            switchTab('login');
        }
    }

    // 🔥 监听 ai_history 变化（任何变化都刷新）
    if (areaName === 'local' && changes.ai_history) {
        console.log('[panel] ai_history 变化，刷新列表');
        chrome.storage.local.get({ ai_history: [] }, r => {
            currentItems = r.ai_history || [];
            renderHistory(currentItems);
        });
    }
});

chrome.storage.local.get({ ai_history: [], active_tasks: {}, user_info: null, token: null, preferred_model: 'aliyun', _openLoginModal: false }, (res) => {
    console.log('[初始化] 页面开始加载, res:', res);

    currentItems = res.ai_history || [];
    renderHistory(currentItems);

    const tasks = res.active_tasks || {};
    const now = Date.now();
    let hasChanges = false;
    Object.keys(tasks).forEach(taskId => {
        const task = tasks[taskId];
        if (!task.startTime || (now - task.startTime > 60000)) {
            if (task.status === 'loading') { updateTaskCard(taskId, 'error', '请求超时'); }
            delete tasks[taskId];
            hasChanges = true;
        } else {
            updateTaskCard(taskId, task.status, task.text);
        }
    });
    if (hasChanges) chrome.storage.local.set({ 'active_tasks': tasks });

    updateHeaderUI(res.user_info, res.token);

    if (modelSelect) {
        if (res.user_info && res.user_info.isVip) { modelSelect.disabled = false; modelSelect.value = res.preferred_model; }
        else { modelSelect.disabled = true; modelSelect.value = 'aliyun'; }
    }

    if (res.token) {
        fetch(`${API_BASE}/user/me`, { headers: { 'Authorization': `Bearer ${res.token}` } })
        .then(r => {
            if (r.status === 401) {
                chrome.storage.local.remove(['user_info', 'token']);
                updateHeaderUI(null, null);
                showToast("登录已过期，请重新登录", "warning");
                return null;
            }
            return r.ok ? r.json() : null;
        })
        .then(data => {
            if (data && data.user) {
                // 🔥 始终更新UI（确保积分等最新状态被加载）
                chrome.storage.local.set({ user_info: data.user });
                updateHeaderUI(data.user, res.token);
            }
        }).catch(() => {});
    }

    // 🔥 检查是否需要打开登录框（处理页面加载时标志位已存在的情况）
    if (res._openLoginModal === true) {
        chrome.storage.local.remove('_openLoginModal');
        if (authModal) {
            authModal.classList.remove('hidden');
            switchTab('login');
        }
    }
});

if (modelSelect) {
    modelSelect.addEventListener('change', () => {
        chrome.storage.local.set({ preferred_model: modelSelect.value });
    });
}

// === 2. 消息监听 ===
chrome.runtime.onMessage.addListener((msg) => {
    if (msg.action === "taskUpdate") {
        updateTaskCard(msg.taskId, msg.status, msg.data);
        if (msg.status === "success") {
            setTimeout(() => {
                chrome.storage.local.get({ ai_history: [] }, r => {
                    currentItems = r.ai_history || [];
                    renderHistory(currentItems);
                });
            }, 100);
        }
    }
    // 处理来自 content script 的全屏弹窗保存请求
    if (msg.action === "saveCloudRecord") {
        handleSaveCloudRecordFromContent(msg);
    }
    // 刷新云端记录列表
    if (msg.action === "refreshCloudRecords") {
        loadCloudRecords();
    }
    // 处理来自二创弹窗的保存请求
    if (msg.action === "saveRewriteChanges") {
        rewriteContent.innerHTML = `<span class="char-watermark">${msg.content.length}字</span>${msg.content}`;
        showToast("二创结果已更新", "success");

        // 🔥 同步更新 ai_history 中的改写记录（最新一条）
        updateRewriteHistory(msg.content);
    }
    // 刷新提取记录列表（来自独立窗口保存后）
    if (msg.action === "refreshExtractionRecords") {
        currentItems = msg.items || [];
        renderHistory(currentItems);
    }
    // 处理二创弹窗关闭
    if (msg.action === "rewriteModalWindowClosed") {
        rewriteModalWindowData = null;
    }
});

// 处理来自 content script 的保存请求
async function handleSaveCloudRecordFromContent(msg) {
    const { recordId, originalText, rewriteText } = msg;
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) return;

    try {
        const res = await fetch(`${API_BASE}/user/rewrite/update/${recordId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ originalText, rewriteText })
        });

        if (res.ok) {
            // 发送成功响应回 content script
            chrome.runtime.sendMessage({
                action: 'saveCloudRecordResponse',
                success: true,
                recordId: recordId
            });
            await loadCloudRecords(); // 刷新列表
        } else {
            const data = await res.json();
            chrome.runtime.sendMessage({
                action: 'saveCloudRecordResponse',
                success: false,
                error: data.error || "保存失败"
            });
        }
    } catch (e) {
        console.error(e);
        chrome.runtime.sendMessage({
            action: 'saveCloudRecordResponse',
            success: false,
            error: "网络错误"
        });
    }
}

function updateTaskCard(taskId, status, text) {
    if (!activeTaskList) return;
    let card = document.getElementById(taskId);
    if (!card && (status === 'loading' || status === 'error')) {
        card = document.createElement('div'); card.id = taskId; card.className = 'loading-card';
        card.innerHTML = `
            <div class="item-header">
                <div class="info-group"><span class="status-badge">Processing</span></div>
                <div class="close-task-btn" title="关闭">×</div>
            </div>
            <div class="loading-placeholder">
                <div class="spinner-mini"></div>
                <span class="loading-text">${text}</span>
            </div>
        `;
        const removeFn = () => { card.remove(); chrome.storage.local.get({ active_tasks: {} }, r => { delete r.active_tasks[taskId]; chrome.storage.local.set({active_tasks: r.active_tasks}); }); };
        card.querySelector('.close-task-btn').addEventListener('click', removeFn);
        activeTaskList.prepend(card);
    }
    if (status === 'loading') {
        if(card) {
            card.classList.remove('error');
            const textEl = card.querySelector('.loading-text');

            // 渐出 -> 换文字 -> 渐入
            textEl.classList.add('fade-out');
            setTimeout(() => {
                textEl.innerText = text;
                textEl.classList.remove('fade-out');
                textEl.classList.add('fade-in');
                setTimeout(() => {
                    textEl.classList.remove('fade-in');
                }, 400);
            }, 400);

            card.querySelector('.status-badge').innerText = "AI 转化中...";
            card.querySelector('.status-badge').className = 'status-badge';
            const spinner = card.querySelector('.spinner-mini');
            if(spinner) spinner.style.display = 'block';
        }
    } else if (status === 'success') {
        if (card) card.remove();
        // 🔥 只有当有实际内容时才显示成功提示（额度超限时 text 为空，不应显示成功）
        if (text && text.trim() !== "") {
            showToast("提取成功", "success");
        }
    } else if (status === 'error') {
        if (card) {
            card.className = 'loading-card error';
            card.querySelector('.loading-text').innerText = text;
            card.querySelector('.status-badge').innerText = "失败";
            card.querySelector('.status-badge').classList.add('error');
            const spinner = card.querySelector('.spinner-mini');
            if(spinner) spinner.style.display = 'none';
            const closeBtn = card.querySelector('.close-task-btn');
            closeBtn.replaceWith(closeBtn.cloneNode(true));
            card.querySelector('.close-task-btn').addEventListener('click', () => {
                card.remove();
                chrome.storage.local.get({ active_tasks: {} }, r => { delete r.active_tasks[taskId]; chrome.storage.local.set({active_tasks: r.active_tasks}); });
            });
        }
    }
}

// === 3. 登录/注册逻辑 ===
btnDoLogin.addEventListener('click', async () => {
    const u = loginUsername.value.trim(), p = loginPassword.value.trim();
    showAuthMsg(loginMsg, "", "default");
    if (!u || !p) { showAuthMsg(loginMsg, "请输入用户名和密码", "error"); return; }

    btnDoLogin.classList.add('btn-loading'); btnDoLogin.disabled = true;
    try {
        const res = await fetch(`${API_BASE}/auth/login`, {
            method: 'POST', headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ username: u, password: p })
        });
        const data = await res.json();
        if (res.ok) {
            chrome.storage.local.set({ user_info: data.user, token: data.token }, () => {
                updateHeaderUI(data.user, data.token);
                authModal.classList.add('hidden');
                loginUsername.value = ""; loginPassword.value = "";
                showToast("欢迎回来," + data.user.username, "success");
            });
        } else { showAuthMsg(loginMsg, data.error || "登录失败", "error"); }
    } catch (e) { showAuthMsg(loginMsg, "网络连接失败", "error"); }
    finally { btnDoLogin.classList.remove('btn-loading'); btnDoLogin.disabled = false; }
});

// 🔥 发送验证码逻辑
btnSendCode.addEventListener('click', async () => {
    const phone = regPhone.value.trim();
    if (!/^1[3-9]\d{9}$/.test(phone)) {
        showAuthMsg(regMsg, "请输入正确的手机号", "error");
        return;
    }

    btnSendCode.disabled = true;
    btnSendCode.classList.add('countdown');
    let count = 60;
    btnSendCode.innerText = `${count}s`;

    const timer = setInterval(() => {
        count--;
        btnSendCode.innerText = `${count}s`;
        if (count <= 0) {
            clearInterval(timer);
            btnSendCode.innerText = "获取验证码";
            btnSendCode.classList.remove('countdown');
            btnSendCode.disabled = false;
        }
    }, 1000);

    try {
        const res = await fetch(`${API_BASE}/auth/send-code`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ phone })
        });
        const data = await res.json();
        if (res.ok) {
            showToast("验证码已发送", "success");
        } else {
            showAuthMsg(regMsg, data.error || "发送失败", "error");
            clearInterval(timer);
            btnSendCode.innerText = "获取验证码";
            btnSendCode.classList.remove('countdown');
            btnSendCode.disabled = false;
        }
    } catch (e) {
        showAuthMsg(regMsg, "网络错误", "error");
        clearInterval(timer);
        btnSendCode.innerText = "获取验证码";
        btnSendCode.classList.remove('countdown');
        btnSendCode.disabled = false;
    }
});

// 🔥 注册逻辑 (包含手机号+验证码)
btnDoRegister.addEventListener('click', async () => {
    const u = regUsername.value.trim();
    const p = regPassword.value.trim();
    const phone = regPhone.value.trim();
    const code = regCode.value.trim();

    showAuthMsg(regMsg, "", "default");

    if (!validateInput(regUsername, errorUsername, (v) => v.length >= 4 && usernameRegex.test(v), "需英文/数字，至少4位")) return;
    if (!validateInput(regPassword, errorPassword, (v) => v.length >= 6, "密码长度至少6位")) return;
    if (!phone || !code) { showAuthMsg(regMsg, "请填写手机号和验证码", "error"); return; }

    btnDoRegister.classList.add('btn-loading'); btnDoRegister.disabled = true;
    try {
        const res = await fetch(`${API_BASE}/auth/register`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ username: u, password: p, phone, code })
        });
        const data = await res.json();
        if (res.ok) {
            showAuthMsg(regMsg, "🎉 注册成功！即将跳转...", "success");
            setTimeout(() => { switchTab('login'); loginUsername.value = u; loginPassword.focus(); }, 1500);
        } else { showAuthMsg(regMsg, data.error || "注册失败", "error"); }
    } catch (e) { showAuthMsg(regMsg, "网络连接失败", "error"); }
    finally { btnDoRegister.classList.remove('btn-loading'); btnDoRegister.disabled = false; }
});

authTabs.forEach(t => t.addEventListener('click', () => switchTab(t.dataset.tab)));
function switchTab(type) {
    authTabs.forEach(t => t.classList.remove('active'));
    document.querySelector(`.auth-tab[data-tab="${type}"]`).classList.add('active');

    showAuthMsg(loginMsg, "", "default");
    showAuthMsg(regMsg, "", "default");

    [regUsername, regPassword].forEach(i => {
        i.value = ""; i.classList.remove('input-valid', 'input-invalid');
        const wrapper = i.parentElement;
        const err = wrapper.querySelector('.field-error');
        if(err) err.innerText = "";
        const icon = wrapper.querySelector('.valid-icon');
        if(icon) icon.classList.add('hidden');
    });

    if (type === 'login') { tabLogin.classList.remove('hidden'); tabRegister.classList.add('hidden'); }
    else { tabLogin.classList.add('hidden'); tabRegister.classList.remove('hidden'); }
}

regUsername.addEventListener('input', () => validateInput(regUsername, errorUsername, (v) => v.length >= 4 && usernameRegex.test(v), "需英文/数字，至少4位"));
regPassword.addEventListener('input', () => validateInput(regPassword, errorPassword, (v) => v.length >= 6, "密码长度至少6位"));

// ... (以下部分保持不变，为节省空间已省略，请确保原文件底部的 AI 逻辑、历史记录逻辑都在) ...
// (请确保你不仅复制了上面的，还保留了 panel.js 中 performAiRewrite, renderHistory 等后续函数)
// 为防止误删，我把剩下的核心逻辑也补在这里，请直接全选覆盖：

// === 保存改写结果到历史记录 ===
async function saveRewriteToHistory(originalContent, rewriteText) {
    const { ai_history } = await new Promise(r => chrome.storage.local.get(['ai_history'], r));
    const items = ai_history || [];

    // 查找匹配的历史项（内容匹配）
    const index = items.findIndex(item =>
        item.content === originalContent || item.content === originalContent.trim()
    );

    if (index !== -1) {
        // 初始化为数组（如果还不是数组）
        if (!items[index].rewrite || !Array.isArray(items[index].rewrite)) {
            items[index].rewrite = [];
        }

        // 追加新的改写记录
        const now = new Date();
        const timeStr = `${(now.getMonth()+1).toString().padStart(2,'0')}-${now.getDate().toString().padStart(2,'0')} ${now.getHours().toString().padStart(2,'0')}:${now.getMinutes().toString().padStart(2,'0')}`;

        items[index].rewrite.push({
            text: rewriteText,
            time: timeStr
        });

        await new Promise(r => chrome.storage.local.set({ ai_history: items }, r));
        renderHistory(items); // 刷新显示
    }
}

// === 更新改写历史记录中的最新一条 ===
async function updateRewriteHistory(newRewriteText) {
    const { ai_history } = await new Promise(r => chrome.storage.local.get(['ai_history'], r));
    const items = ai_history || [];

    // 查找匹配的历史项（通过 currentRewriteOriginalText）
    const index = items.findIndex(item =>
        item.content === currentRewriteOriginalText || item.content === (currentRewriteOriginalText || '').trim()
    );

    if (index !== -1 && items[index].rewrite && Array.isArray(items[index].rewrite) && items[index].rewrite.length > 0) {
        // 更新最新一条改写记录（数组最后一个元素）
        const lastRecordIndex = items[index].rewrite.length - 1;
        const now = new Date();
        const timeStr = `${(now.getMonth()+1).toString().padStart(2,'0')}-${now.getDate().toString().padStart(2,'0')} ${now.getHours().toString().padStart(2,'0')}:${now.getMinutes().toString().padStart(2,'0')}`;

        items[index].rewrite[lastRecordIndex] = {
            text: newRewriteText,
            time: timeStr
        };

        await new Promise(r => chrome.storage.local.set({ ai_history: items }, r));
        renderHistory(items); // 刷新显示
    }
}

// === 查看已保存的改写结果 ===
function showRewriteResult(rewriteText, originalText = '') {
    aiHeader.classList.remove('hidden');
    rewriteContent.innerHTML = `<span class="char-watermark">${rewriteText.length}字</span>${rewriteText}`;
    currentRewriteOriginalText = originalText; // 保存原文以便云端保存
    showToast("已加载改写结果", "success");
}

async function performAiRewrite(sourceText) {
    if (!sourceText) { showToast("文案内容为空！", "warning"); return; }
    const { token, user_info } = await new Promise(r => chrome.storage.local.get(['token', 'user_info'], r));
    if (!token) { showToast("AI 二创功能需要登录后使用", "warning"); authModal.classList.remove('hidden'); switchTab('login'); return; }

    // 🔥 检查 VIP 状态 - 在发起请求之前直接拦截
    if (user_info && !user_info.isVip) {
        rewriteContent.innerText = "AI 爆款改写是 VIP 专属功能，升级后即可使用";
        showToast("升级 VIP 使用此功能", "warning");
        return;
    }

    aiHeader.classList.remove('hidden');
    rewriteContent.innerHTML = `
        <div class="ai-loading-container">
            <div class="wave-bars">
                <div class="wave-bar"></div>
                <div class="wave-bar"></div>
                <div class="wave-bar"></div>
                <div class="wave-bar"></div>
                <div class="wave-bar"></div>
            </div>
            <div class="ai-loading-text">🧠 AI 正在拆解爆款逻辑...</div>
        </div>
    `;

    try {
        const res = await fetch(`${API_BASE}/rewrite`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${token}` },
            body: JSON.stringify({ text: sourceText, type: 'rewrite' })
        });
        const data = await res.json();

        if (res.ok) {
            const rewriteResult = data.result;
            // 移除动画类，强制重排后再添加以触发动画
            rewriteContent.classList.remove('fade-in');
            void rewriteContent.offsetWidth; // 强制重排
            rewriteContent.innerHTML = `<span class="char-watermark">${rewriteResult.length}字</span>${rewriteResult}`;
            rewriteContent.classList.add('fade-in');
            // 保存原文以便云端保存时使用
            currentRewriteOriginalText = sourceText;
            showToast("分析完成！", "success");

            // 🔥 保存改写结果到历史记录
            await saveRewriteToHistory(sourceText, rewriteResult);
        }
        else {
            // 🔥 处理 401 登录过期
            if (res.status === 401) {
                showToast("登录已过期，请重新登录", "error");
                chrome.storage.local.remove(['user_info', 'token']);
                updateHeaderUI(null, null);
                return;
            }

            // 🔥 处理 403 VIP 权限
            if (res.status === 403) {
                rewriteContent.innerText = (data.message || "升级 VIP 即可使用 AI 智能改写功能");
                showToast(data.error || "VIP专属功能", "error");
                // 可以在这里添加打开支付页面的逻辑
                return;
            }

            rewriteContent.innerText = "❌ 失败: " + (data.error || "未知错误");
            showToast(data.error || "AI 生成失败", "error");
        }
    } catch (e) { rewriteContent.innerText = "❌ 网络错误，请检查网络或稍后重试。"; }
}

// === 修复：AI 爆款重写区域点击放大编辑 ===
rewriteContent.addEventListener('click', () => {
    // 1. 如果是初始提示语（空状态），不给点
    if(rewriteContent.querySelector('.empty-ai-tip')) return;

    // 获取纯文本，移除可能存在的水印元素
    const watermarks = rewriteContent.querySelectorAll('.char-watermark');
    let text = rewriteContent.innerText;
    // 移除水印文字（如"289字"）
    watermarks.forEach(wm => {
        text = text.replace(wm.textContent, '').trim();
    });

    // 2. 只有当明确是"正在加载"的提示时才禁止点击
    if(text.includes('正在分析爆款逻辑') || text.includes('AI 正在思考')) {
        return;
    }

    // 3. 打开全屏弹窗编辑
    openRewriteModal(text);
});

btnCopyRewrite.addEventListener('click', () => {
    // 获取纯文本，移除可能存在的水印元素
    const watermarks = rewriteContent.querySelectorAll('.char-watermark');
    let text = rewriteContent.innerText;
    // 移除水印文字（如"289字"）
    watermarks.forEach(wm => {
        text = text.replace(wm.textContent, '').trim();
    });

    if(!text || text.includes('...')) return;
    navigator.clipboard.writeText(text).then(() => {
        showToast("已复制到剪贴板", "success");
    });
});

// === 保存到云端按钮 ===
btnSaveRewriteCloud.addEventListener('click', async () => {
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) { showToast("请先登录", "warning"); return; }

    // 获取纯文本，移除可能存在的水印元素
    const watermarks = rewriteContent.querySelectorAll('.char-watermark');
    let rewriteText = rewriteContent.innerText;
    // 移除水印文字（如"289字"）
    watermarks.forEach(wm => {
        rewriteText = rewriteText.replace(wm.textContent, '').trim();
    });

    // 检查是否有有效内容
    if (!rewriteText || rewriteText.includes('...') || rewriteText.includes('正在拆解爆款逻辑') || rewriteContent.querySelector('.empty-ai-tip')) {
        showToast("没有可保存的内容", "warning");
        return;
    }

    // 显示加载状态
    const originalText = btnSaveRewriteCloud.innerText;
    btnSaveRewriteCloud.innerText = "保存中...";
    btnSaveRewriteCloud.disabled = true;

    try {
        const res = await fetch(`${API_BASE}/user/rewrite/save`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                originalText: currentRewriteOriginalText || '',
                rewriteText: rewriteText
            })
        });

        if (res.ok) {
            showToast("已保存到云端", "success");
            btnSaveRewriteCloud.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"/></svg> 已保存';
            setTimeout(() => btnSaveRewriteCloud.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg> 保存到云端', 3000);
        } else {
            const data = await res.json();
            showToast(data.error || "保存失败", "error");
            btnSaveRewriteCloud.innerText = originalText;
        }
    } catch (e) {
        console.error(e);
        showToast("网络错误，保存失败", "error");
        btnSaveRewriteCloud.innerText = originalText;
    } finally {
        btnSaveRewriteCloud.disabled = false;
    }
});

function renderHistory(items, isSearchMode = false) {
    if(!historyList) return;
    historyList.innerHTML = "";
    if (!items || items.length === 0) {
        historyList.innerHTML = `<div class="empty-tip">${isSearchMode ? '未找到相关文案' : '暂无提取记录'}</div>`;
        return;
    }

    // 存储所有记录（仅在非搜索模式下）
    if (!isSearchMode) {
        allHistoryItems = [...items];
    }

    // 更新搜索框状态
    if (searchInput && clearSearchBtn) {
        if (searchInput.value) {
            searchInput.parentElement.classList.add('has-value');
        } else {
            searchInput.parentElement.classList.remove('has-value');
        }
    }
    [...items].reverse().forEach((item, index) => {
        let content = typeof item === 'string' ? item : item.content;
        let timeStr = typeof item === 'string' ? "历史" : (item.time || "");
        const originalIndex = (items.length - 1) - index;
        const div = document.createElement('div'); div.className = 'history-item';

        // 🔥 判断是否是警告状态
        const isWarningState = deleteWarningIndex === originalIndex;

        div.innerHTML = `
            <div class="item-header">
                <div class="info-group">
                    <span class="item-num">#${originalIndex + 1}</span>
                    <span class="item-time">${timeStr}</span>
                </div>
                <div class="action-group">
                    <button class="action-btn item-rewrite-btn">AI改写</button>
                    <button class="action-btn item-copy-btn" title="复制">
                        <svg xmlns="http://www.w3.org/2000/svg" width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;"><rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect><path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path></svg>
                    </button>
                    <button class="action-btn item-del-btn ${isWarningState ? 'warning' : ''}" title="${isWarningState ? '确认删除' : '删除此记录'}">
                        ${isWarningState
                            ? '<svg xmlns="http://www.w3.org/2000/svg" width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;vertical-align:middle;margin-right:2px;"><line x1="12" y1="9" x2="12" y2="13"></line><line x1="12" y1="17" x2="12.01" y2="17"></line><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path></svg> 确认'
                            : '<svg xmlns="http://www.w3.org/2000/svg" width="13" height="13" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;vertical-align:middle;"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg>'
                        }
                    </button>
                </div>
                </div>
            <div class="item-text" title="点击展开">
                <span class="char-watermark">${content.length}字</span>${content}
            </div>`;

        // 🔥 AI改写按钮 - 始终显示"AI改写"，不再显示"查看改写"
        // 兼容新旧格式：数组格式或对象格式
        const hasRewrite = item.rewrite && (Array.isArray(item.rewrite) ? item.rewrite.length > 0 : !!item.rewrite.text);
        const btnRewrite = div.querySelector('.item-rewrite-btn');
        btnRewrite.innerText = 'AI改写';
        btnRewrite.classList.remove('has-rewrite');
        btnRewrite.addEventListener('click', (e) => {
            e.stopPropagation();
            performAiRewrite(content);
        });

        // 🔥 在内容旁边添加改写标记
        if (hasRewrite) {
            const badge = document.createElement('span');
            badge.className = 'rewrite-badge';
            badge.innerText = '已改写';
            badge.style.cursor = 'pointer';  // 添加指针样式
            badge.title = '点击查看改写结果';  // 添加悬停提示
            div.querySelector('.info-group').appendChild(badge);

            // 🔥 添加点击事件：显示改写历史弹窗
            badge.addEventListener('click', (e) => {
                e.stopPropagation();
                showRewriteHistoryModal(content, item.rewrite);
            });
        }

        // 🔥 删除按钮 - 两次点击确认机制
        const delBtn = div.querySelector('.item-del-btn');
        delBtn.addEventListener('click', (e) => {
            e.stopPropagation();

            if (deleteWarningIndex !== originalIndex) {
                // 第一次点击：进入警告状态
                deleteWarningIndex = originalIndex;
                renderHistory(items, isSearchMode);

                // 3秒后自动重置警告状态
                setTimeout(() => {
                    if (deleteWarningIndex === originalIndex) {
                        deleteWarningIndex = -1;
                        renderHistory(items, isSearchMode);
                    }
                }, 3000);
            } else {
                // 第二次点击：执行删除
                deleteWarningIndex = -1;
                items.splice(originalIndex, 1);
                chrome.storage.local.set({ ai_history: items });
                renderHistory(items, isSearchMode);
                showToast("已删除", "default");
            }
        });

        const copyBtn = div.querySelector('.item-copy-btn');
        copyBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            navigator.clipboard.writeText(content).then(() => showToast("已复制", "success"));
        });
        // 点击文字区域打开独立全屏编辑弹窗
        div.querySelector('.item-text').addEventListener('click', (e) => {
            e.stopPropagation();
            openExtractionModal(originalIndex, content, items);
        });
        historyList.appendChild(div);
    });
}

// === 云端记录功能 ===

// 打开云端记录弹窗
if (openCloudModalBtn) {
    openCloudModalBtn.addEventListener('click', async () => {
        const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
        if (!token) {
            showToast("请先登录", "warning");
            return;
        }

        cloudModal.classList.remove('hidden');
        loadCloudRecords();
    });
}

// 关闭云端记录弹窗
if (closeCloudModalBtn) {
    closeCloudModalBtn.addEventListener('click', () => {
        cloudModal.classList.add('hidden');
    });
}

// ==================== 分屏弹窗功能 ====================

// 关闭分屏弹窗
function closeSplitModal() {
    splitModal.classList.add('hidden');
    currentSplitRecordId = null;
}

if (closeSplitModalBtn) {
    closeSplitModalBtn.addEventListener('click', closeSplitModal);
}

if (splitCancelBtn) {
    splitCancelBtn.addEventListener('click', closeSplitModal);
}

// 打开分屏弹窗 - 使用独立窗口（不依赖抖音网页）
function openSplitModal(record) {
    currentSplitRecordId = record._id;

    // 创建独立的全屏窗口
    chrome.windows.create({
        url: chrome.runtime.getURL('split-modal.html'),
        type: 'popup',
        width: 1400,
        height: 850,
        focused: true
    }, (window) => {
        if (chrome.runtime.lastError) {
            console.error('创建窗口失败:', chrome.runtime.lastError);
            // 回退到 panel 内弹窗
            splitOriginalText.value = record.originalText || '';
            splitRewriteText.value = record.rewriteText || '';
            splitModal.classList.remove('hidden');
            return;
        }

        // 存储当前记录数据，供新窗口初始化使用
        splitModalWindowData = {
            windowId: window.id,
            record: record
        };

        // 等待窗口加载完成后初始化数据
        const initTimeout = setTimeout(() => {
            console.warn('初始化超时，使用备用方案');
        }, 5000);

        // 监听窗口准备就绪消息
        const messageListener = (request, sender, sendResponse) => {
            if (request.action === 'splitModalWindowReady') {
                clearTimeout(initTimeout);
                // 发送初始化数据到新窗口
                chrome.tabs.query({ windowId: window.id }, (tabs) => {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: 'initSplitModal',
                            data: record
                        });
                    }
                });
                chrome.runtime.onMessage.removeListener(messageListener);
            }
        };
        chrome.runtime.onMessage.addListener(messageListener);
    });
}

// 打开提取记录编辑弹窗 - 使用独立窗口
function openExtractionModal(index, content, items) {
    // 创建独立的全屏窗口
    chrome.windows.create({
        url: chrome.runtime.getURL('extraction-modal.html'),
        type: 'popup',
        width: 900,
        height: 700,
        focused: true
    }, (window) => {
        if (chrome.runtime.lastError) {
            console.error('创建窗口失败:', chrome.runtime.lastError);
            // 回退到 panel 内弹窗
            isEditingRewrite = false;
            currentEditIndex = index;
            currentItems = items;
            largeEditorArea.value = content;
            editModal.classList.remove('hidden');
            return;
        }

        // 存储当前记录数据，供新窗口初始化使用
        extractionModalWindowData = {
            windowId: window.id,
            index: index,
            content: content,
            items: items
        };

        // 等待窗口加载完成后初始化数据
        const initTimeout = setTimeout(() => {
            console.warn('初始化超时，使用备用方案');
        }, 5000);

        // 监听窗口准备就绪消息
        const messageListener = (request, sender, sendResponse) => {
            if (request.action === 'extractionModalWindowReady') {
                clearTimeout(initTimeout);
                // 发送初始化数据到新窗口
                chrome.tabs.query({ windowId: window.id }, (tabs) => {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: 'initExtractionModal',
                            data: {
                                index: index,
                                content: content
                            }
                        });
                    }
                });
                chrome.runtime.onMessage.removeListener(messageListener);
            }
        };
        chrome.runtime.onMessage.addListener(messageListener);
    });
}

// 打开二创结果编辑弹窗 - 使用独立窗口
function openRewriteModal(content) {
    // 创建独立的全屏窗口
    chrome.windows.create({
        url: chrome.runtime.getURL('rewrite-modal.html'),
        type: 'popup',
        width: 1200,
        height: 700,
        focused: true
    }, (window) => {
        if (chrome.runtime.lastError) {
            console.error('创建窗口失败:', chrome.runtime.lastError);
            // 回退到 panel 内弹窗
            isEditingRewrite = true;
            largeEditorArea.value = content;
            editModal.classList.remove('hidden');
            return;
        }

        // 存储当前记录数据
        rewriteModalWindowData = {
            windowId: window.id,
            content: content
        };

        // 等待窗口加载完成后初始化数据
        const initTimeout = setTimeout(() => {
            console.warn('初始化超时，使用备用方案');
        }, 5000);

        // 监听窗口准备就绪消息
        const messageListener = (request, sender, sendResponse) => {
            if (request.action === 'rewriteModalWindowReady') {
                clearTimeout(initTimeout);
                chrome.tabs.query({ windowId: window.id }, (tabs) => {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            action: 'initRewriteModal',
                            data: {
                                originalText: currentRewriteOriginalText || '',
                                content: content
                            }
                        });
                    }
                });
                chrome.runtime.onMessage.removeListener(messageListener);
            }
        };
        chrome.runtime.onMessage.addListener(messageListener);
    });
}

// 全局变量存储分屏窗口数据
let splitModalWindowData = null;

// 复制原文按钮
if (splitCopyOriginal) {
    splitCopyOriginal.addEventListener('click', () => {
        navigator.clipboard.writeText(splitOriginalText.value).then(() => {
            showToast("原文已复制", "success");
        });
    });
}

// 复制二创按钮
if (splitCopyRewrite) {
    splitCopyRewrite.addEventListener('click', () => {
        navigator.clipboard.writeText(splitRewriteText.value).then(() => {
            showToast("二创已复制", "success");
        });
    });
}

// 保存分屏修改按钮
if (splitSaveBtn) {
    splitSaveBtn.addEventListener('click', async () => {
        const newOriginalText = splitOriginalText.value.trim();
        const newRewriteText = splitRewriteText.value.trim();

        if (!newRewriteText) {
            showToast("二创内容不能为空", "warning");
            return;
        }

        const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
        if (!token) {
            showToast("请先登录", "warning");
            return;
        }

        try {
            const res = await fetch(`${API_BASE}/user/rewrite/update/${currentSplitRecordId}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    originalText: newOriginalText,
                    rewriteText: newRewriteText
                })
            });

            if (res.ok) {
                showToast("保存成功", "success");
                closeSplitModal();
                await loadCloudRecords(); // 重新加载列表
            } else {
                const data = await res.json();
                showToast(data.error || "保存失败", "error");
            }
        } catch (e) {
            console.error(e);
            showToast("网络错误", "error");
        }
    });
}

// 加载云端记录
async function loadCloudRecords() {
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) {
        if (cloudModalList) {
            cloudModalList.innerHTML = '<div class="empty-tip">请先登录查看云端记录</div>';
        }
        return;
    }

    if (cloudModalList) {
        cloudModalList.innerHTML = '<div class="empty-tip">加载中...</div>';
    }

    try {
        const res = await fetch(`${API_BASE}/user/rewrite/history`, {
            headers: { 'Authorization': `Bearer ${token}` }
        });

        if (res.ok) {
            const data = await res.json();
            currentCloudRecords = data.history || [];
            renderCloudRecords(currentCloudRecords);
        } else {
            if (cloudModalList) {
                cloudModalList.innerHTML = '<div class="empty-tip">加载失败，请稍后重试</div>';
            }
        }
    } catch (e) {
        console.error(e);
        if (cloudModalList) {
            cloudModalList.innerHTML = '<div class="empty-tip">网络错误</div>';
        }
    }
}

// 渲染云端记录列表
function renderCloudRecords(records) {
    if (!cloudModalList) return;
    cloudModalList.innerHTML = "";

    if (!records || records.length === 0) {
        cloudModalList.innerHTML = '<div class="empty-tip">暂无云端记录<br><small style="color:#999;">保存AI改写记录后可在此查看</small></div>';
        return;
    }

    records.forEach((item, index) => {
        const div = document.createElement('div');
        div.className = 'cloud-modal-item';

        const date = new Date(item.createdAt);
        const dateStr = `${date.getMonth() + 1}-${date.getDate()} ${date.getHours().toString().padStart(2, '0')}:${date.getMinutes().toString().padStart(2, '0')}`;

        // 构建按钮HTML，如果存在原文则添加"原文"按钮
        let buttonsHtml = '';
        if (item.originalText) {
            buttonsHtml += `<button class="cloud-original-btn" data-index="${index}">原文</button>`;
        }
        buttonsHtml += `<button class="cloud-copy-btn" data-index="${index}">复制</button>`;
        buttonsHtml += `<button class="cloud-delete-btn" data-index="${index}">删除</button>`;

        div.innerHTML = `
            <div class="cloud-item-header">
                <span class="cloud-item-time">${dateStr}</span>
                <div class="cloud-item-actions">
                    ${buttonsHtml}
                </div>
            </div>
            <div class="cloud-item-text"><span class="char-watermark">${item.rewriteText.length}字</span>${item.rewriteText}</div>
        `;

        // 原文按钮 - 打开原文查看弹窗
        if (item.originalText) {
            div.querySelector('.cloud-original-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                showOriginalTextModal(item.originalText);
            });
        }

        // 复制按钮
        div.querySelector('.cloud-copy-btn').addEventListener('click', (e) => {
            e.stopPropagation();
            navigator.clipboard.writeText(item.rewriteText).then(() => {
                showToast("已复制", "success");
            });
        });

        // 删除按钮
        div.querySelector('.cloud-delete-btn').addEventListener('click', (e) => {
            e.stopPropagation();
            showConfirm("确定要删除这条记录吗？", async () => {
                await deleteCloudRecord(item._id);
            });
        });

        // 点击整个卡片打开分屏弹窗（包括文字区域）
        div.addEventListener('click', (e) => {
            if (e.target.tagName === 'BUTTON') return;
            openSplitModal(item);
        });

        cloudModalList.appendChild(div);
    });
}

// 显示原文查看弹窗
function showOriginalTextModal(originalText) {
    // 创建临时弹窗
    const modal = document.createElement('div');
    modal.className = 'modal show';
    modal.style.zIndex = '10003';
    modal.innerHTML = `
        <div class="modal-content large-editor">
            <div class="modal-header">
                <h3>原文</h3>
                <span class="close-btn modal-close">×</span>
            </div>
            <div class="original-text-content"><span class="char-watermark">${originalText.length}字</span>${originalText}</div>
        </div>
    `;

    document.body.appendChild(modal);

    // 关闭弹窗
    const closeBtn = modal.querySelector('.modal-close');
    const closeModal = () => {
        modal.remove();
    };
    closeBtn.addEventListener('click', closeModal);
    modal.addEventListener('click', (e) => {
        if (e.target === modal) closeModal();
    });
}

// 显示改写历史弹窗 - 使用独立窗口
function showRewriteHistoryModal(originalText, rewriteData) {
    // 🔥 从最新的 storage 中获取数据，而不是使用传入的旧数据
    chrome.storage.local.get(['ai_history'], (res) => {
        const items = res.ai_history || [];

        // 🔥 查找匹配的历史项（标准化比较，去除首尾空格）
        const normalizedOriginalText = (originalText || '').trim();
        const matchedItem = items.find(item => {
            const itemContent = (typeof item === 'string' ? item : (item.content || '')).trim();
            return itemContent === normalizedOriginalText;
        });

        // 🔥 获取最新的改写数据（检查是否为空数组）
        let latestRewriteData = rewriteData; // 默认使用传入的数据
        if (matchedItem) {
            if (matchedItem.rewrite) {
                if (Array.isArray(matchedItem.rewrite) && matchedItem.rewrite.length > 0) {
                    latestRewriteData = matchedItem.rewrite;
                } else if (!Array.isArray(matchedItem.rewrite) && matchedItem.rewrite.text) {
                    latestRewriteData = [matchedItem.rewrite];
                }
            }
        }

        console.log('[showRewriteHistoryModal] 匹配项:', matchedItem ? '找到' : '未找到', '改写记录:', latestRewriteData);

        // 存储数据供新窗口使用
        rewriteHistoryWindowData = {
            originalText: originalText,
            rewriteData: latestRewriteData
        };

        // 创建独立窗口
        chrome.windows.create({
            url: chrome.runtime.getURL('rewrite-history-modal.html'),
            type: 'popup',
            width: 900,
            height: 700,
            focused: true
        }, (window) => {
            if (chrome.runtime.lastError) {
                console.error('创建窗口失败:', chrome.runtime.lastError);
                showToast('打开窗口失败', 'error');
                return;
            }

            // 监听窗口准备就绪消息
            const messageListener = (request, sender, sendResponse) => {
                if (request.action === 'rewriteHistoryWindowReady') {
                    // 发送初始化数据到新窗口
                    chrome.tabs.query({ windowId: window.id }, (tabs) => {
                        if (tabs[0]) {
                            chrome.tabs.sendMessage(tabs[0].id, {
                                action: 'initRewriteHistoryModal',
                                data: rewriteHistoryWindowData
                            });
                        }
                    });
                    chrome.runtime.onMessage.removeListener(messageListener);
                }
            };
            chrome.runtime.onMessage.addListener(messageListener);
        });
    });
}

// 删除云端记录
async function deleteCloudRecord(recordId) {
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) return;

    try {
        const res = await fetch(`${API_BASE}/user/rewrite/delete/${recordId}`, {
            method: 'DELETE',
            headers: { 'Authorization': `Bearer ${token}` }
        });

        if (res.ok) {
            showToast("已删除", "success");
            await loadCloudRecords(); // 重新加载列表
        } else {
            const data = await res.json();
            showToast(data.error || "删除失败", "error");
        }
    } catch (e) {
        console.error(e);
        showToast("网络错误", "error");
    }
}

// 更新云端记录
async function updateCloudRecord(recordId, newText) {
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) return false;

    try {
        const res = await fetch(`${API_BASE}/user/rewrite/update/${recordId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ rewriteText: newText })
        });

        if (res.ok) {
            showToast("已保存", "success");
            await loadCloudRecords(); // 重新加载列表
            return true;
        } else {
            const data = await res.json();
            showToast(data.error || "保存失败", "error");
            return false;
        }
    } catch (e) {
        console.error(e);
        showToast("网络错误", "error");
        return false;
    }
}

btnEditSave.addEventListener('click', async () => {
    const newContent = largeEditorArea.value.trim();
    if (!newContent) { showToast("内容不能为空", "warning"); return; }

    // 🔥 检查是否在编辑云端记录
    if (currentEditCloudIndex >= 0 && currentCloudRecords[currentEditCloudIndex]) {
        const record = currentCloudRecords[currentEditCloudIndex];
        const success = await updateCloudRecord(record._id, newContent);
        if (success) {
            editModal.classList.add('hidden');
            currentEditCloudIndex = -1;
        }
        return;
    }

    // 原有的本地记录编辑逻辑
    if (isEditingRewrite) {
        rewriteContent.innerText = newContent;
        showToast("AI 结果已更新", "success");
    } else {
        if (typeof currentItems[currentEditIndex] === 'string') {
            currentItems[currentEditIndex] = newContent;
        } else {
            currentItems[currentEditIndex].content = newContent;
        }
        chrome.storage.local.set({ ai_history: currentItems });
        renderHistory(currentItems);
        showToast("记录已保存", "success");
    }
    editModal.classList.add('hidden');
    isEditingRewrite = false;
});
btnEditCancel.addEventListener('click', () => editModal.classList.add('hidden'));
closeEditBtn.addEventListener('click', () => editModal.classList.add('hidden'));

notLoggedInDiv.addEventListener('click', () => { authModal.classList.remove('hidden'); switchTab('login'); });
closeAuthBtn.addEventListener('click', () => authModal.classList.add('hidden'));
closeProfileBtn.addEventListener('click', () => profileModal.classList.add('hidden'));

if(goVipBtn) goVipBtn.addEventListener('click', async () => {
    chrome.storage.local.get(['token'], async (res) => {
        if (res.token) {
            // 请求临时票据
            try {
                const response = await fetch('https://zhushou.xokx.top/api/auth/ticket', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${res.token}`
                    },
                    body: JSON.stringify({ type: 'vip' })
                });

                if (response.ok) {
                    const data = await response.json();
                    // 使用ticket打开VIP支付页面
                    chrome.tabs.create({
                        url: `https://zhushou.xokx.top/pages/pay.html#ticket=${data.ticket}`
                    });
                } else {
                    showToast("获取票据失败，请重试", "error");
                }
            } catch (e) {
                console.error('[获取票据] 错误:', e);
                showToast("网络错误，请重试", "error");
            }
        } else {
            showToast("请先登录", "warning");
            authModal.classList.remove('hidden');
            switchTab('login');
        }
    });
});
if(goProfileBtn) goProfileBtn.addEventListener('click', async () => {
    chrome.storage.local.get(['user_info', 'token'], async res => {
        if(res.user_info) {
            // 🔥 如果是VIP用户，先刷新积分余额
            if (res.user_info.isVip && res.token) {
                try {
                    await loadPointsBalance();
                } catch (e) {
                    console.log('[刷新积分] 失败:', e);
                }
            }

            // 🔥 获取最新的用户信息
            const freshRes = await fetch('https://zhushou.xokx.top/api/user/me', {
                headers: { 'Authorization': `Bearer ${res.token}` }
            });
            if (freshRes.ok) {
                const data = await freshRes.json();
                if (data.user) {
                    // 更新本地存储的用户信息
                    chrome.storage.local.set({ user_info: data.user });
                    res.user_info = data.user;
                }
            }

            // 显示用户信息
            document.getElementById('display-name').innerText = res.user_info.username;
            profileAvatarImg.src = "https://zhushou.xokx.top/assets/tx.png";
            if(res.user_info.isVip) {
                profileBadge.innerText = "VIP 会员";
                profileBadge.className = "badge pro";
                profileExpireText.innerText = getExpireText(res.user_info.vipExpireAt);
            } else {
                profileBadge.innerText = "免费版";
                profileBadge.className = "badge free";
                profileExpireText.innerText = "升级解锁无限提取";
            }
            profileModal.classList.remove('hidden');
        }
    });
});
if(menuLogoutBtn) menuLogoutBtn.addEventListener('click', () => {
    showConfirm("确定要退出当前账号吗？", () => {
        chrome.storage.local.remove(['user_info', 'token'], () => {
            updateHeaderUI(null, null);
            showToast("已退出", "default");
        });
    });
});
if(clearBtn) clearBtn.addEventListener('click', () => {
    showConfirm("确定要清空所有提取记录吗？", () => {
        chrome.storage.local.set({ ai_history: [] });
        renderHistory([]);
        allHistoryItems = [];
        showToast("列表已清空", "default");
    });
});

// 🔥 搜索功能
if (searchInput) {
    searchInput.addEventListener('input', (e) => {
        const keyword = e.target.value.trim();
        if (keyword) {
            const filteredItems = allHistoryItems.filter(item => {
                const content = typeof item === 'string' ? item : item.content;
                return content.toLowerCase().includes(keyword.toLowerCase());
            });
            renderHistory(filteredItems, true); // true 表示搜索模式
        } else {
            renderHistory(allHistoryItems, false); // 空搜索显示全部
        }
    });
}

if (clearSearchBtn) {
    clearSearchBtn.addEventListener('click', () => {
        searchInput.value = '';
        searchInput.parentElement.classList.remove('has-value');
        renderHistory(allHistoryItems, false);
        searchInput.focus();
    });
}
if (logoutBtn) {
    logoutBtn.addEventListener('click', () => {
        // 直接执行退出逻辑，不再调用 showConfirm
        chrome.storage.local.remove(['user_info', 'token'], () => {
            updateHeaderUI(null, null);
            profileModal.classList.add('hidden'); // 关闭个人中心
            showToast("已退出", "success");
        });
    });
}
if (openFolderBtn) openFolderBtn.addEventListener('click', () => chrome.downloads.showDefaultFolder());

// 菜单链接事件监听（使用说明和隐私政策）
if (linkManual) {
    linkManual.addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ url: 'https://zhushou.xokx.top/pages/manual.html' });
    });
}

if (linkPrivacy) {
    linkPrivacy.addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ url: 'https://zhushou.xokx.top/pages/privacy.html' });
    });
}

function formatDate(dateStr) { if (!dateStr) return ""; const d = new Date(dateStr); const year = d.getFullYear(); const month = (d.getMonth() + 1).toString().padStart(2, '0'); const day = d.getDate().toString().padStart(2, '0'); return `${year}-${month}-${day}`; }
function getExpireText(expireAt) { if (!expireAt) return "有效期: 长期"; const d = new Date(expireAt); if (d.getFullYear() > 2090) return "有效期: 永久"; return `有效期至: ${formatDate(expireAt)}`; }

// 🔥 加载积分余额
async function loadPointsBalance() {
    console.log('[积分] 开始加载积分...');
    const { token } = await new Promise(r => chrome.storage.local.get(['token'], r));
    if (!token) {
        console.log('[积分] 未找到 token');
        return;
    }

    try {
        const res = await fetch('https://zhushou.xokx.top/api/points/balance', {
            headers: { 'Authorization': `Bearer ${token}` }
        });
        const data = await res.json();

        console.log('[积分] API 返回数据:', data);

        if (data.success !== false) {
            console.log('[积分] 余额:', data.balance, '月度上限:', data.monthlyLimit);

            // 🔥 更新 UI 显示积分余额
            const pointsBalanceEl = document.getElementById('points-balance');
            const pointsMonthlyEl = document.getElementById('points-monthly');
            const pointsContainerEl = document.getElementById('points-balance-container');

            console.log('[积分] 元素:', { pointsBalanceEl, pointsMonthlyEl, pointsContainerEl });

            if (pointsBalanceEl) pointsBalanceEl.textContent = data.balance;
            if (pointsMonthlyEl) pointsMonthlyEl.textContent = data.monthlyLimit;
            if (pointsContainerEl) {
                pointsContainerEl.style.display = 'inline';
                console.log('[积分] 积分容器已显示');
            }
        } else {
            console.log('[积分] API 返回 success=false');
        }
    } catch (e) {
        console.log('[积分] 获取余额失败:', e);
    }
}

// === 通知功能 ===
const NOTICE_STORAGE_KEY = 'last_notice_id';
const NOTICE_LATER_COUNT_KEY = 'notice_later_counts';
const MAX_LATER_COUNT = 3; // 最多显示3次"稍后通知"
let currentNoticeId = null;

// 图标 SVG
const NOTICE_ICONS = {
    info: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/></svg>`,
    success: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>`,
    warning: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/><line x1="12" y1="9" x2="12" y2="13"/><line x1="12" y1="17" x2="12.01" y2="17"/></svg>`,
    error: `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>`
};

/**
 * 获取并显示通知（仅普通通知，已移除版本更新）
 */
async function fetchAndShowNotice() {
    fetchRegularNotice();
}

/**
 * 获取普通通知
 */
async function fetchRegularNotice() {
    try {
        const response = await fetch('https://zhushou.xokx.top/api/plugin/notice');
        const data = await response.json();

        if (data.success && data.notice && data.notice.enabled !== false) {
            const notice = data.notice;
            const noticeId = `notice_${notice.title}_${notice.updatedAt || Date.now()}`;

            // 检查通知状态（是否已被永久关闭或稍后次数已用完）
            chrome.storage.local.get([NOTICE_STORAGE_KEY, NOTICE_LATER_COUNT_KEY], (result) => {
                const lastNoticeId = result[NOTICE_STORAGE_KEY];
                const laterCounts = result[NOTICE_LATER_COUNT_KEY] || {};

                // 如果是新的通知，或"稍后通知"次数未用完
                if (lastNoticeId !== noticeId) {
                    showNotice(notice, noticeId);
                } else if (laterCounts[noticeId] !== undefined && laterCounts[noticeId] < MAX_LATER_COUNT) {
                    // 用户点击过"稍后通知"，继续显示
                    showNotice(notice, noticeId);
                }
            });
        }
    } catch (error) {
        console.log('[通知] 获取通知失败:', error);
    }
}

/**
 * 显示普通通知
 */
function showNotice(notice, noticeId) {
    const modal = document.getElementById('notice-modal');
    const iconEl = document.getElementById('notice-icon');
    const titleEl = document.getElementById('notice-title');
    const contentEl = document.getElementById('notice-content');
    const btnLater = document.getElementById('btn-notice-later');
    const btnNever = document.getElementById('btn-notice-never');

    // 设置通知内容和图标
    const iconType = notice.type || 'info';
    iconEl.innerHTML = NOTICE_ICONS[iconType] || NOTICE_ICONS.info;
    iconEl.className = `notice-modal-icon icon-${iconType}`;

    titleEl.textContent = notice.title;
    contentEl.textContent = notice.content;

    // 恢复按钮文字
    btnLater.textContent = '稍后通知';
    btnNever.textContent = '永不通知';

    // 绑定按钮事件
    btnLater.onclick = () => dismissNoticeWithLater(noticeId);
    btnNever.onclick = () => dismissNoticeForever(noticeId);

    // 显示弹窗
    modal.classList.add('show');
    currentNoticeId = noticeId;
}

/**
 * 关闭通知弹窗
 */
function dismissNoticeModal() {
    const modal = document.getElementById('notice-modal');
    modal.classList.remove('show');
    currentNoticeId = null;
}

/**
 * 稍后通知：增加计数，下次还会弹
 */
function dismissNoticeWithLater(noticeId) {
    chrome.storage.local.get([NOTICE_LATER_COUNT_KEY], (result) => {
        const laterCounts = result[NOTICE_LATER_COUNT_KEY] || {};
        const currentCount = laterCounts[noticeId] || 0;

        // 增加计数
        laterCounts[noticeId] = currentCount + 1;

        // 保存计数
        chrome.storage.local.set({
            [NOTICE_LATER_COUNT_KEY]: laterCounts,
            [NOTICE_STORAGE_KEY]: noticeId
        });

        dismissNoticeModal();
    });
}

/**
 * 永不通知：标记为永久关闭
 */
function dismissNoticeForever(noticeId) {
    chrome.storage.local.get([NOTICE_LATER_COUNT_KEY], (result) => {
        const laterCounts = result[NOTICE_LATER_COUNT_KEY] || {};

        // 设置为最大值，表示不再显示
        laterCounts[noticeId] = MAX_LATER_COUNT;

        // 保存
        chrome.storage.local.set({
            [NOTICE_LATER_COUNT_KEY]: laterCounts,
            [NOTICE_STORAGE_KEY]: noticeId
        });

        dismissNoticeModal();
    });
}

// 页面加载时获取通知
fetchAndShowNotice();

// 可选：定期检查通知（每小时）
const noticeIntervalId = setInterval(fetchAndShowNotice, 60 * 60 * 1000);

// 页面卸载时清理定时器，防止内存泄漏
window.addEventListener('beforeunload', () => {
    clearInterval(noticeIntervalId);
});
const DEFAULT_AVATAR = "https://zhushou.xokx.top/assets/tx.png";

function updateHeaderUI(user, token) {
    const badge = document.querySelector('.brand .badge');
    console.log('[updateHeaderUI] 用户信息:', { user, token });

    if (user && token) {
        // 已登录状态
        notLoggedInDiv.classList.add('hidden');
        loggedInDiv.classList.remove('hidden');
        avatarImg.src = DEFAULT_AVATAR;
        menuUsername.innerText = user.username;

        if (user.isVip) {
            console.log('[updateHeaderUI] 是VIP用户，加载积分');
            // VIP 用户
            badge.innerText = "PRO";
            badge.className = "badge pro";
            menuRole.innerText = "VIP 会员";
            menuRole.style.color = "#d97706";
            vipExpireDateEl.innerText = getExpireText(user.vipExpireAt);
            vipExpireDateEl.style.display = 'block';

            // 🔥 加载积分余额
            loadPointsBalance();

            if (modelSelect) {
                modelSelect.disabled = false;
                chrome.storage.local.get(['preferred_model'], r => {
                    modelSelect.value = r.preferred_model || 'aliyun';
                });
            }
        } else {
            // 免费用户
            badge.innerText = "FREE";
            badge.className = "badge free";
            menuRole.innerText = "免费版";
            menuRole.style.color = "#999";
            vipExpireDateEl.style.display = 'none';

            if (modelSelect) {
                modelSelect.disabled = true;
                modelSelect.value = 'aliyun';
            }
        }
    } else {
        // 未登录状态
        notLoggedInDiv.classList.remove('hidden');
        loggedInDiv.classList.add('hidden');
        avatarImg.src = "";
        badge.innerText = "FREE";
        badge.className = "badge free";

        if (modelSelect) {
            modelSelect.disabled = true;
            modelSelect.value = 'aliyun';
        }
    }
}

// 🔥 积分说明浮窗功能
document.addEventListener('DOMContentLoaded', function() {
    const pointsHelpBtn = document.getElementById('points-help-btn');
    const pointsInfoModal = document.getElementById('points-info-modal');
    const closePointsInfo = document.getElementById('close-points-info');

    // 点击问号图标显示说明浮窗
    if (pointsHelpBtn) {
        pointsHelpBtn.addEventListener('click', function(e) {
            e.stopPropagation();
            if (pointsInfoModal) {
                pointsInfoModal.style.display = 'flex';
            }
        });
    }

    // 点击关闭按钮隐藏浮窗
    if (closePointsInfo) {
        closePointsInfo.addEventListener('click', function() {
            if (pointsInfoModal) {
                pointsInfoModal.style.display = 'none';
            }
        });
    }

    // 点击浮窗外部区域关闭浮窗
    if (pointsInfoModal) {
        pointsInfoModal.addEventListener('click', function(e) {
            if (e.target === pointsInfoModal) {
                pointsInfoModal.style.display = 'none';
            }
        });
    }

    // 🔥 鼠标悬停头像时刷新积分余额
    const userMenuWrapper = document.querySelector('.user-menu-wrapper');
    if (userMenuWrapper) {
        userMenuWrapper.addEventListener('mouseenter', async function() {
            // 只对VIP用户刷新积分
            chrome.storage.local.get(['user_info', 'token'], async res => {
                if (res.user_info && res.user_info.isVip && res.token) {
                    try {
                        await loadPointsBalance();
                    } catch (e) {
                        console.log('[悬停刷新积分] 失败:', e);
                    }
                }
            });
        });
    }
});

// 🔥 监听充值成功事件（通过localStorage通信）
window.addEventListener('storage', function(e) {
    if (e.key === 'zimeizhushou_refresh_points' && e.newValue) {
        console.log('[面板刷新] 检测到充值成功，开始刷新积分');

        // 刷新积分余额
        loadPointsBalance().then(() => {
            console.log('[面板刷新] 积分刷新完成');
        }).catch(err => {
            console.log('[面板刷新] 刷新积分失败:', err);
        });
    }
});

// 🔥 积分充值按钮点击事件
const pointsRechargeBtn = document.getElementById('points-recharge-btn');
if (pointsRechargeBtn) {
    pointsRechargeBtn.addEventListener('click', async function() {
        chrome.storage.local.get(['token'], async (res) => {
            if (res.token) {
                // 请求临时票据
                try {
                    const response = await fetch('https://zhushou.xokx.top/api/auth/ticket', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'Authorization': `Bearer ${res.token}`
                        },
                        body: JSON.stringify({ type: 'recharge' })
                    });

                    if (response.ok) {
                        const data = await response.json();
                        // 使用ticket打开充值页面
                        chrome.tabs.create({ url: `https://zhushou.xokx.top/pages/recharge.html#ticket=${data.ticket}` });
                    } else {
                        showToast("获取票据失败，请重试", "error");
                    }
                } catch (e) {
                    console.error('[获取票据] 错误:', e);
                    showToast("网络错误，请重试", "error");
                }
            } else {
                showToast("请先登录", "warning");
                authModal.classList.remove('hidden');
                switchTab('login');
            }
        });
    });
}
