// rewrite-history-modal.js - 改写历史窗口逻辑

let originalText = '';
let rewriteRecords = [];
let editingIndex = -1; // 当前正在编辑的索引，-1 表示没有在编辑
let deleteWarningIndex = -1; // 删除警告状态索引，-1 表示没有警告状态

// 显示 Toast 提示
function showToast(message) {
    const oldToast = document.querySelector('.global-toast');
    if (oldToast) oldToast.remove();

    const toast = document.createElement('div');
    toast.className = 'global-toast success';
    toast.textContent = message;
    document.body.appendChild(toast);

    // 触发重排后添加 show 类
    requestAnimationFrame(() => {
        toast.classList.add('show');
    });

    setTimeout(() => {
        toast.classList.remove('show');
        toast.classList.add('hiding');
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

// 监听来自 panel 的初始化消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'initRewriteHistoryModal') {
        const { originalText: origText, rewriteData } = request.data;

        originalText = origText;

        // 处理旧数据格式（单个对象）和新数据格式（数组）
        if (Array.isArray(rewriteData)) {
            rewriteRecords = rewriteData;
        } else if (rewriteData && rewriteData.text) {
            // 旧格式：单个对象，转换为数组
            rewriteRecords = [rewriteData];
        } else {
            rewriteRecords = [];
        }

        // 渲染界面
        renderUI();

        // 关闭按钮事件
        document.getElementById('close-btn').addEventListener('click', () => {
            window.close();
        });

        sendResponse({ received: true });
    }
    return true;
});

// 渲染界面
function renderUI() {
    // 显示原文
    document.getElementById('original-text').textContent = originalText || '无原文';

    // 更新原文字数统计
    document.getElementById('originalCharCount').textContent = `${(originalText || '无原文').length}字`;

    // 更新记录数量
    const countEl = document.getElementById('record-count');
    countEl.textContent = `(${rewriteRecords.length}条)`;

    // 渲染改写记录列表
    const listEl = document.getElementById('rewrite-list');
    listEl.innerHTML = '';

    if (rewriteRecords.length === 0) {
        listEl.innerHTML = '<div class="empty-rewrite-tip">暂无改写记录</div>';
        return;
    }

    // 倒序显示（最新在最前面）
    [...rewriteRecords].reverse().forEach((record, index) => {
        const actualIndex = rewriteRecords.length - 1 - index;
        const isEditing = actualIndex === editingIndex;

        const item = document.createElement('div');
        item.className = `rewrite-list-item ${isEditing ? 'editing' : ''}`;

        if (isEditing) {
            // 编辑模式
            item.innerHTML = `
                <div class="rewrite-item-header">
                    <span class="rewrite-item-title"><svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="vertical-align: middle; margin-right: 4px;"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/><path d="M18 2l4 4-10 10-4 1 1-4 10-10z"/></svg>${record.time}</span>
                    <span class="rewrite-item-badge">编辑中</span>
                </div>
                <textarea class="rewrite-edit-textarea" placeholder="请输入改写内容...">${record.text || ''}</textarea>
                <div class="rewrite-edit-actions">
                    <button class="action-btn copy-btn" data-index="${actualIndex}">
                        <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="9" y="9" width="13" height="13" rx="2" ry="2"/><path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"/></svg>
                        复制
                    </button>
                    <button class="action-btn save-btn" data-index="${actualIndex}">
                        <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg>
                        保存
                    </button>
                    <button class="action-btn cancel-btn" data-index="${actualIndex}">
                        <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>
                        取消
                    </button>
                </div>
            `;

            // 先添加到DOM，再调整高度（关键：元素必须在DOM中才能正确计算scrollHeight）
            listEl.appendChild(item);

            // 绑定按钮事件
            const textarea = item.querySelector('.rewrite-edit-textarea');

            // 动态调整高度，确保没有滚动条
            const adjustHeight = () => {
                // 第1步：移除所有限制，强制重排
                textarea.style.height = 'auto';
                textarea.style.minHeight = 'auto';
                textarea.style.overflowY = 'hidden';
                textarea.style.overflowX = 'hidden';

                // 第2步：强制浏览器重排
                textarea.scrollTop = textarea.scrollTop;

                // 第3步：在下一帧设置为 visible 并计算
                requestAnimationFrame(() => {
                    textarea.style.overflowY = 'visible';
                    // 再等一帧确保 DOM 完全稳定
                    requestAnimationFrame(() => {
                        const scrollHeight = textarea.scrollHeight;
                        const newHeight = scrollHeight + 10;  // 更大的缓冲值

                        // 同时设置所有属性，避免中间状态
                        textarea.style.height = newHeight + 'px';
                        textarea.style.minHeight = newHeight + 'px';
                        textarea.style.overflowY = 'hidden';
                    });
                });
            };

            // 初始化高度
            requestAnimationFrame(adjustHeight);

            // 监听输入事件，实时调整高度
            textarea.addEventListener('input', adjustHeight);

            textarea.focus();

            // 复制按钮
            item.querySelector('.copy-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                navigator.clipboard.writeText(textarea.value).then(() => {
                    showToast('已复制到剪贴板');
                }).catch(() => {
                    showToast('复制失败');
                });
            });

            // 保存按钮
            item.querySelector('.save-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                saveEdit(actualIndex, textarea.value);
            });

            // 取消按钮
            item.querySelector('.cancel-btn').addEventListener('click', (e) => {
                e.stopPropagation();
                editingIndex = -1;
                renderUI();
            });

        } else {
            // 只读模式 - 只有第一条显示"最新"标签
            const isWarningState = actualIndex === deleteWarningIndex;
            item.innerHTML = `
                <div class="rewrite-item-header">
                    <span class="rewrite-item-title">${record.time}</span>
                    <div class="rewrite-item-actions">
                        ${index === 0 ? '<span class="rewrite-item-badge">最新</span>' : ''}
                        <button class="action-btn delete-btn ${isWarningState ? 'warning' : ''}" data-index="${actualIndex}" title="${isWarningState ? '确认删除' : '删除此记录'}">
                            ${isWarningState
                                ? '<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;"><line x1="12" y1="9" x2="12" y2="13"></line><line x1="12" y1="17" x2="12.01" y2="17"></line><path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path></svg> 确认'
                                : '<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="flex-shrink:0;"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg>'
                            }
                        </button>
                    </div>
                </div>
                <div class="rewrite-item-content" title="点击编辑"><span class="char-watermark">${(record.text || '').length}字</span>${record.text || ''}</div>
            `;

            // 只有点击内容区域才进入编辑模式
            const contentEl = item.querySelector('.rewrite-item-content');
            contentEl.addEventListener('click', () => {
                editingIndex = actualIndex;
                renderUI();
            });

            // 删除按钮事件 - 两次点击机制
            const deleteBtn = item.querySelector('.delete-btn');
            deleteBtn.addEventListener('click', (e) => {
                e.stopPropagation();

                // 第一次点击：进入警告状态
                if (deleteWarningIndex !== actualIndex) {
                    deleteWarningIndex = actualIndex;
                    renderUI(); // 重新渲染显示警告状态

                    // 3秒后自动重置警告状态
                    setTimeout(() => {
                        if (deleteWarningIndex === actualIndex) {
                            deleteWarningIndex = -1;
                            renderUI();
                        }
                    }, 3000);
                } else {
                    // 第二次点击：执行删除
                    deleteWarningIndex = -1;
                    deleteRecord(actualIndex);
                }
            });
        }

        // 只读模式，在末尾添加到DOM
        listEl.appendChild(item);
    });
}

// 保存编辑
function saveEdit(index, newContent) {
    const trimmedContent = newContent.trim();

    if (!trimmedContent) {
        showToast('内容不能为空');
        return;
    }

    // 更新本地记录
    rewriteRecords[index] = {
        text: trimmedContent,
        time: rewriteRecords[index].time
    };

    // 发送保存请求到 background
    chrome.runtime.sendMessage({
        action: 'saveRewriteHistoryRecord',
        originalText: originalText,
        newRewriteText: trimmedContent,
        recordIndex: index
    }, (response) => {
        if (chrome.runtime.lastError) {
            showToast('保存失败');
            return;
        }

        if (response && response.success) {
            showToast('保存成功');
            editingIndex = -1;
            renderUI();
        } else {
            showToast('保存失败');
        }
    });
}

// 删除记录
function deleteRecord(index) {
    // 从本地数组中删除
    rewriteRecords.splice(index, 1);

    // 更新 chrome.storage.local 中的数据
    chrome.storage.local.get(['ai_history'], (res) => {
        const items = res.ai_history || [];

        // 查找匹配的历史项（标准化比较）
        const normalizedOriginalText = (originalText || '').trim();
        const historyIndex = items.findIndex(item => {
            const itemContent = (typeof item === 'string' ? item : (item.content || '')).trim();
            return itemContent === normalizedOriginalText;
        });

        if (historyIndex !== -1) {
            // 确保 rewrite 是数组
            if (!items[historyIndex].rewrite) {
                items[historyIndex].rewrite = [];
            } else if (!Array.isArray(items[historyIndex].rewrite)) {
                // 如果是旧格式（对象），转换为数组
                items[historyIndex].rewrite = [items[historyIndex].rewrite];
            }

            // 删除对应索引的改写记录
            items[historyIndex].rewrite.splice(index, 1);

            // 如果改写记录数组为空，删除整个 rewrite 属性
            if (items[historyIndex].rewrite.length === 0) {
                delete items[historyIndex].rewrite;
            }

            // 保存更新后的数据
            chrome.storage.local.set({ ai_history: items }, () => {
                if (chrome.runtime.lastError) {
                    showToast('删除失败');
                    // 恢复本地数据
                    renderUI();
                } else {
                    showToast('删除成功');
                    // 重新渲染，如果正在编辑被删除的记录，重置编辑状态
                    if (editingIndex === index) {
                        editingIndex = -1;
                    } else if (editingIndex > index) {
                        editingIndex--;
                    }
                    renderUI();
                    // chrome.storage.onChanged 会自动触发 panel.js 的刷新
                }
            });
        } else {
            showToast('删除失败：记录不存在');
            renderUI();
        }
    });
}

// 窗口加载完成后通知 background
window.addEventListener('load', () => {
    chrome.runtime.sendMessage({
        action: 'rewriteHistoryWindowReady'
    });
});
