// rewrite-modal.js - 二创结果编辑窗口逻辑（分屏模式）

let rewriteModalData = null;

// DOM 元素
const closeBtn = document.getElementById('closeBtn');
const cancelBtn = document.getElementById('cancelBtn');
const saveBtn = document.getElementById('saveBtn');
const copyOriginalBtn = document.getElementById('copyOriginalBtn');
const copyRewriteBtn = document.getElementById('copyRewriteBtn');
const originalText = document.getElementById('originalText');
const rewriteText = document.getElementById('rewriteText');
const originalCharCount = document.getElementById('originalCharCount');
const rewriteCharCount = document.getElementById('rewriteCharCount');

// 更新字数统计
function updateCharCount() {
    originalCharCount.textContent = `${originalText.value.length}字`;
    rewriteCharCount.textContent = `${rewriteText.value.length}字`;
}

// 初始化
function init() {
    // 通知父窗口已准备好
    chrome.runtime.sendMessage({
        action: 'rewriteModalWindowReady'
    });

    // 监听来自父窗口的消息
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.action === 'initRewriteModal') {
            const { originalText: origText, content } = request.data;
            originalText.value = origText || '无原文';
            rewriteText.value = content || '';
            rewriteModalData = { originalText: origText, content };
            updateCharCount();
        }
    });
}

// 关闭窗口
function closeWindow() {
    chrome.runtime.sendMessage({
        action: 'rewriteModalWindowClosed'
    });
    window.close();
}

// 保存修改
function saveChanges() {
    const newContent = rewriteText.value.trim();
    if (!newContent) {
        showToast('二创内容不能为空');
        return;
    }

    chrome.runtime.sendMessage({
        action: 'saveRewriteChanges',
        content: newContent
    });

    showToast('已保存');
    setTimeout(() => closeWindow(), 500);
}

// 复制原文
function copyOriginal() {
    const text = originalText.value;
    if (!text || text === '无原文') return;

    navigator.clipboard.writeText(text).then(() => {
        copyOriginalBtn.textContent = '已复制';
        setTimeout(() => {
            copyOriginalBtn.textContent = '复制';
        }, 2000);
    });
}

// 复制二创结果
function copyRewrite() {
    const text = rewriteText.value;
    if (!text) return;

    navigator.clipboard.writeText(text).then(() => {
        copyRewriteBtn.textContent = '已复制';
        setTimeout(() => {
            copyRewriteBtn.textContent = '复制';
        }, 2000);
    });
}

// 显示 Toast 提示
function showToast(message) {
    const existingToast = document.querySelector('.dy-copy-toast');
    if (existingToast) {
        existingToast.remove();
    }

    const toast = document.createElement('div');
    toast.className = 'dy-copy-toast';
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => {
        toast.classList.add('hiding');
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

// 事件监听
closeBtn.addEventListener('click', closeWindow);
cancelBtn.addEventListener('click', closeWindow);
saveBtn.addEventListener('click', saveChanges);
copyOriginalBtn.addEventListener('click', copyOriginal);
copyRewriteBtn.addEventListener('click', copyRewrite);

// 监听输入事件，实时更新字数
originalText.addEventListener('input', updateCharCount);
rewriteText.addEventListener('input', updateCharCount);

// ESC 键关闭
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        closeWindow();
    }
});

// Ctrl+S 保存
document.addEventListener('keydown', (e) => {
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        saveChanges();
    }
});

// 页面加载完成后初始化
init();
