// split-modal.js - 独立窗口的分屏弹窗逻辑

let currentRecordId = null;
let currentRecordData = null;

// 获取 DOM 元素
const modalOverlay = document.getElementById('modalOverlay');
const originalText = document.getElementById('originalText');
const rewriteText = document.getElementById('rewriteText');
const closeBtn = document.getElementById('closeBtn');
const cancelBtn = document.getElementById('cancelBtn');
const saveBtn = document.getElementById('saveBtn');
const copyBtns = document.querySelectorAll('.dy-split-copy');
const originalCharCount = document.getElementById('originalCharCount');
const rewriteCharCount = document.getElementById('rewriteCharCount');

// 更新字数统计
function updateCharCount() {
    originalCharCount.textContent = `${originalText.value.length}字`;
    rewriteCharCount.textContent = `${rewriteText.value.length}字`;
}

// 显示 Toast 提示
function showToast(message) {
    const oldToast = document.querySelector('.dy-copy-toast');
    if (oldToast) oldToast.remove();

    const toast = document.createElement('div');
    toast.className = 'dy-copy-toast';
    toast.textContent = message;
    document.body.appendChild(toast);

    setTimeout(() => {
        toast.classList.add('hiding');
        setTimeout(() => toast.remove(), 300);
    }, 2000);
}

// 初始化弹窗数据
function initModal(data) {
    currentRecordId = data._id;
    currentRecordData = data;
    originalText.value = data.originalText || '';
    rewriteText.value = data.rewriteText || '';
    updateCharCount();
}

// 关闭窗口
function closeModal() {
    window.close();
}

// 绑定事件
closeBtn.addEventListener('click', closeModal);
cancelBtn.addEventListener('click', closeModal);

modalOverlay.addEventListener('click', (e) => {
    if (e.target === modalOverlay) closeModal();
});

// 监听输入事件，实时更新字数
originalText.addEventListener('input', updateCharCount);
rewriteText.addEventListener('input', updateCharCount);

// 复制按钮
copyBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        const type = btn.dataset.type;
        const textarea = type === 'original' ? originalText : rewriteText;
        navigator.clipboard.writeText(textarea.value).then(() => {
            showToast(type === 'original' ? '原文已复制' : '二创已复制');
        });
    });
});

// 保存按钮
saveBtn.addEventListener('click', async () => {
    const newOriginalText = originalText.value.trim();
    const newRewriteText = rewriteText.value.trim();

    if (!newRewriteText) {
        showToast('二创内容不能为空');
        return;
    }

    // 发送保存请求到 background.js
    chrome.runtime.sendMessage({
        action: 'saveCloudRecordFromSplitModal',
        recordId: currentRecordId,
        originalText: newOriginalText,
        rewriteText: newRewriteText
    }, (response) => {
        if (chrome.runtime.lastError) {
            showToast('保存失败：' + chrome.runtime.lastError.message);
            return;
        }

        if (response && response.success) {
            showToast('保存成功');
            setTimeout(() => {
                closeModal();
            }, 500);
        } else {
            showToast('保存失败：' + (response?.error || '未知错误'));
        }
    });
});

// 监听来自 panel 的初始化消息
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'initSplitModal') {
        initModal(request.data);
        sendResponse({ received: true });
    }
    return true;
});

// 窗口加载完成后通知 background
window.addEventListener('load', () => {
    // 通知 background 窗口已准备好
    chrome.runtime.sendMessage({
        action: 'splitModalWindowReady'
    });
});

// 窗口关闭前清理
window.addEventListener('beforeunload', () => {
    chrome.runtime.sendMessage({
        action: 'splitModalWindowClosed'
    });
});
